<?php declare(strict_types = 1);

namespace MultihubDropShip\Console;

use Doctrine;
use Exception;
use MultihubDropShip\Model\Sync\SyncProducts;
use Override;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Tracy\Debugger;

#[AsCommand(name: 'multihub-drop-ship:update-all')]
class Update extends Command
{
	public function __construct(
		protected SyncProducts $syncProducts,
	)
	{
		parent::__construct();
	}

	#[Override]
	protected function configure(): void
	{
		$this->setName('multihub-drop-ship:update-all')
			->addOption('clientId', null, InputOption::VALUE_REQUIRED)
			->addOption('service', null, InputOption::VALUE_OPTIONAL)
			->setDescription('Update all products');
	}

	#[Override]
	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		$output->writeln('Multihub DropShip - update all products');

		try {
			Debugger::dispatch();
			ini_set('memory_limit', '3G');

			$clientId = is_numeric($input->getOption('clientId')) ? (int) $input->getOption('clientId') : null;
			if (!$clientId) {
				throw new Exception('Client not found');
			}

			if ($input->getOption('service') === 'allegro') {
				$this->syncProducts->sendAllTo($clientId, 'allegro');
			} else if ($input->getOption('service') === 'kaufland') {
				$this->syncProducts->sendAllTo($clientId, 'kaufland');
			} else {
				$this->syncProducts->updateAll($clientId);
			}

			$output->writeLn('<info>[OK] - products updated</info>');

			return 0;
		} catch (Exception $e) {
			Debugger::log($e, 'multihub-update-error');
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 1;
	}
}
