<?php declare(strict_types = 1);

namespace MultihubDropShip\Console;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use Doctrine;
use Exception;
use MultihubDropShip\Model\Api\ProductsApi;
use MultihubDropShip\Model\Clients;
use MultihubDropShip\Model\EshopProducts;
use MultihubDropShip\Model\ProductBuilder;
use Override;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Tracy\Debugger;

#[AsCommand(name: 'multihub-drop-ship:update-from-list')]
class UpdateFromList extends Command
{
	public function __construct(
		protected ProductsApi            $productsApi,
		protected EntityManagerDecorator $em,
		protected Clients                $clients,
		protected EshopProducts          $eshopProducts,
		protected ProductBuilder         $productBuilder,
	)
	{
		parent::__construct();
	}

	#[Override]
	protected function configure(): void
	{
		$this->setName('multihub-drop-ship:update-from-list')
			->setDescription('Update from db list');
	}

	#[Override]
	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		set_time_limit(60 * 10);
		/** @phpstan-ignore-next-line */
		$tmpFile = DATA_DIR . '/multihubUpdating.txt';

		$lastRun = file_exists($tmpFile) ? (int) file_get_contents($tmpFile) : null;
		$now     = time();

		if ($lastRun && ($now - $lastRun) < 60 * 20) {
			$output->writeln('Multihub DropShip - update - already running');

			return 0;
		}

		file_put_contents($tmpFile, time());

		$output->writeln('Multihub DropShip - update');

		try {
			Debugger::dispatch();
			ini_set('memory_limit', '2G');

			try {
				$ids = [];
				foreach ($this->em->getConnection()->fetchAllAssociative("SELECT product_id FROM multihub_drop_ship__product_update_list") as $row) {
					/** @var array $row */
					$ids[] = (int) $row['product_id'];
				}

				if ($ids !== []) {
					foreach ($this->clients->getClients() as $client) {
						$canRemove = [];
						$update    = [];

						foreach ($ids as $id) {
							if ($this->eshopProducts->isProductAllowed($id, $client)) {
								$update[] = $id;
							} else {
								$canRemove[] = $id;
							}
						}

						if ($update !== []) {
							Sites::$currentIdentOverride = $client->siteIdent;
							Sites::$currentLangOverride  = $client->lang;

							$products = $this->productBuilder->buildProducts($client, $update);

							foreach (array_diff_key(array_flip($update), $products) as $id => $v) {
								$canRemove[] = $id;
							}

							if ($canRemove !== []) {
								$this->em->getConnection()->executeQuery('DELETE FROM multihub_drop_ship__product_update_list WHERE `product_id` IN (' . implode(',', $canRemove) . ')');
							}

							try {
								foreach ($products as $id => $product) {
									$res = $this->productsApi->updateProduct($client, $product);

									if (isset($res['accepted']) || isset($res['productId'])) {
										$this->em->getConnection()->executeQuery('DELETE FROM multihub_drop_ship__product_update_list WHERE `product_id` = ' . $id);
									} else {
										Debugger::log($res, 'multihub-drop-ship-update');
									}
								}
							} catch (\Exception $e) {
							}
						}

						if ($canRemove !== []) {
							$this->em->getConnection()->executeQuery('DELETE FROM multihub_drop_ship__product_update_list WHERE `product_id` IN (' . implode(',', $canRemove) . ')');
						}
					}
				}

				$output->writeLn('<info>[OK] - products updated</info>');
			} catch (Exception $e) {
				$output->writeLn('<info>[ERR] - failed - ' . $e->getMessage() . '</info>');
				Debugger::log($e, 'multihub-drop-ship-update');
			}

			unlink($tmpFile);

			return 0;
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 1;
	}
}
