<?php declare(strict_types = 1);

namespace MultihubDropShip\Model;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use Core\Model\Images\ImagePipe;
use Core\Model\TemplateReader\TemplateTextTypesCollection;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use DynamicModule\FrontModule\Model\Dao\Member;
use DynamicModule\FrontModule\Model\Repository\IGroupsFactory;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\FrontModule\Model\Categories as FrontCategories;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Entities\CategoryTexts;
use MultihubDropShip\Model\Dao\Category;
use MultihubDropShip\Model\Dao\CategoryText;
use MultihubDropShip\Model\Dao\Client;
use Navigations\FrontModule\Model\LinksHelper;
use Navigations\Model\Navigations;
use Nette\Utils\Json;

class CategoryBuilder
{
	public function __construct(
		protected EntityManagerDecorator      $em,
		protected Categories                  $categories,
		protected FrontCategories             $frontCategories,
		protected LinksHelper                 $linksHelper,
		protected Navigations                 $navigations,
		protected ImagePipe                   $imagePipe,
		protected IGroupsFactory              $groupsFactory,
		protected Translator                  $translator,
		protected TemplateTextTypesCollection $templateTextTypesCollection,
		protected ProductsFacade              $productsFacade,
	)
	{
	}

	public function buildCategories(Client $client, array $ids): array
	{
		/** @var Category[] $result */
		$result      = [];
		$categoryIds = [];

		$rootCategory        = $this->categories->getSiteRootId($client->siteIdent);
		$eshopCategories     = $this->frontCategories->getCategories($rootCategory, $client->lang);
		$dynamicModuleGroups = $this->groupsFactory->create('dynamic');
		$baseUrl             = $client->imageBaseUrl;
		$defaultLang         = $this->translator->getDefaultLocale();

		$dynamicIds = [];
		foreach ($ids as $k => $id) {
			if (is_string($id) && Strings::startsWith($id, 'V_')) {
				$dynamicIds[] = (int) Strings::substring($id, 2);
				unset($ids[$k]);
			}
		}

		foreach ($eshopCategories as $eshopCategory) {
			$dynamicModuleMembersIds = EshopCategories::$dynamicModuleMembers[$eshopCategory->getId()] ?? [];
			$dynamicModuleGroupsIds  = EshopCategories::$dynamicModuleGroups[$eshopCategory->getId()] ?? [];

			if (
				!Arrays::contains($ids, $eshopCategory->getId())
				&& empty(array_intersect($dynamicModuleMembersIds, $dynamicIds))
			) {
				continue;
			}

			$categoryParentPath = $eshopCategory->getParentPathStringFlipped();
			$categoryName       = (string) $eshopCategory->name;

			if ($categoryParentPath) {
				$categoryName = $categoryParentPath . ' > ' . $categoryName;
			}

			$categoryId = $eshopCategory->getId();

			$category = new Category(
				(string) $categoryId,
				$categoryName,
			);

			$category->parentId       = $eshopCategory->parentId ? (string) $eshopCategory->parentId : null;
			$category->activeProducts = (int) $eshopCategory->activeProducts;

			if ($eshopCategory->lvl !== null) {
				$category->level = $eshopCategory->lvl;
			}

			$images = [];
			if ($eshopCategory->image) {
				$ext = strtolower(pathinfo($eshopCategory->image, PATHINFO_EXTENSION));

				if ($ext === 'svg') {
					$small  = $category->image;
					$medium = $category->image;
				} else {
					$small = $this->imagePipe->request($eshopCategory->image, '100x100', 'fit', false, false, 'webp');
					$small = '/thumbs' . explode('/thumbs', str_replace(' ', '%20', $small), 2)[1];

					$medium = $this->imagePipe->request($eshopCategory->image, '470x266', 'fit', false, false, 'webp');
					$medium = '/thumbs' . explode('/thumbs', str_replace(' ', '%20', $medium), 2)[1];
				}

				$category->image = str_replace(' ', '%20', $client->imageBaseUrl . $eshopCategory->image);

				$images[] = [
					'default' => $category->image,
					'small'   => $client->imageBaseUrl . $small,
					'medium'  => $client->imageBaseUrl . $medium,
				];
			}

			$category->images = $images;

			$categoryIds[$categoryId]    = $categoryId;
			$result[$category->remoteId] = $category;

			// Virtualni kategorie
			foreach ($dynamicModuleGroupsIds as $dynamicModuleGroupId) {
				foreach ($dynamicModuleGroups->getByIds([$dynamicModuleGroupId], $defaultLang) as $dynamicModuleGroup) {
					foreach ($dynamicModuleGroup->getMembers() as $member) {
						$category = new Category(
							'V_' . $member->getId(),
							$categoryName . ' > ' . ($member->getFieldValue('title') ?: $member->getTitle()),
						);

						$category->type           = 'virtual';
						$category->parentId       = (string) $eshopCategory->getId();
						$category->level          = $eshopCategory->lvl + 1;
						$category->activeProducts = (int) ($member->params['activeProducts'] ?? 0);

						$images       = [];
						$defaultImage = $member->getFieldValue('image')['file'] ?? null;
						if ($defaultImage) {
							$ext = strtolower(pathinfo($defaultImage, PATHINFO_EXTENSION));

							if ($ext === 'svg') {
								$small  = $defaultImage;
								$medium = $defaultImage;
							} else {
								$small = $this->imagePipe->request($defaultImage, '100x100', 'fit', false, false, 'webp');
								$small = '/thumbs' . explode('/thumbs', str_replace(' ', '%20', $small), 2)[1];

								$medium = $this->imagePipe->request($defaultImage, '470x266', 'fit', false, false, 'webp');
								$medium = '/thumbs' . explode('/thumbs', str_replace(' ', '%20', $medium), 2)[1];
							}

							$category->image = str_replace(' ', '%20', $client->imageBaseUrl . $defaultImage);

							$images[] = [
								'default' => $category->image,
								'small'   => $client->imageBaseUrl . $small,
								'medium'  => $client->imageBaseUrl . $medium,
							];
						}

						$category->images = $images;

						$text = $this->getDynamicModuleMemberText($member, $defaultLang, $baseUrl);
						if ($text) {
							$category->texts[] = $text;
						}

						$result[$category->remoteId] = $category;
					}
				}

				foreach ($this->translator->getLocalesWhitelist() ?: [] as $lang) {
					if ($lang === $defaultLang) {
						continue;
					}

					foreach ($dynamicModuleGroups->getByIds($dynamicModuleMembersIds, $lang) as $dynamicModuleGroup) {
						foreach ($dynamicModuleGroup->getMembers() as $member) {
							$tmpId = 'V_' . $member->getId();

							if (isset($result[$tmpId])) {
								$text = $this->getDynamicModuleMemberText($member, $lang, $baseUrl);

								if ($text) {
									$result[$tmpId]->texts[] = $text;
								}
							}
						}
					}
				}
			}
		}

		if (empty($result)) {
			return [];
		}

		// Category texts
		$categoryTexts = [];
		foreach (array_chunk($categoryIds, 250) as $chunk) {
			foreach ($this->em->getRepository(CategoryTexts::class)
				         ->createQueryBuilder('ct')
				         ->select('IDENTITY(ct.id) as category, ct.lang, ct.name, ct.nameH1, ct.shortDescription, ct.description, ct.image')
				         ->where('ct.id IN (:ids)')
				         ->setParameters(new ArrayCollection([new Parameter('ids', $chunk)]))
				         ->getQuery()->getArrayResult() as $row) {
				/** @var array $row */
				$cat = $this->frontCategories->getCategories($rootCategory, $row['lang'])[$row['category']] ?? null;

				$text                   = new CategoryText($row['lang'], (string) $row['name']);
				$text->shortName        = (string) $row['name'];
				$text->url              = $cat->link ?? null;
				$text->nameH1           = $row['nameH1'];
				$text->shortDescription = $row['shortDescription'];
				$text->description      = $row['description'];
				$text->safetyGuidelines = $cat ? $cat->getSafetyWarningText() : null;
				$text->safetyDataSheet  = $cat && $cat->getSafetyWarningImage() ? str_replace(' ', '%20', $cat->getSafetyWarningImage()) : null;

				if ($row['image']) {
					$text->image = str_replace(' ', '%20', $client->imageBaseUrl . $row['image']);
				}

				$categoryTexts[$row['category']][$row['lang']] = $text;
			}
		}

		foreach (array_chunk(array_intersect($categoryIds, array_keys($result)), 250) as $chunk) {
			foreach ($chunk as $id) {
				$result[$id]->texts = array_values($categoryTexts[$result[$id]->remoteId]);
			}
		}

		$this->em->clear();

		return $result;
	}

	protected function getDynamicModuleMemberText(Member $member, string $lang, string $baseUrl): ?CategoryText
	{
		$title = $member->getFieldValueScalar('title');
		if (!$title) {
			return null;
		}

		$text            = new CategoryText($lang, (string) $title);
		$text->shortName = $text->name;
		$text->nameH1    = $text->name;

		$ttLink = $this->templateTextTypesCollection->getItemByType('link');
		$link   = $member->getFieldValue('link') ? Json::decode((string) $member->getFieldValueScalar('link'), forceArrays: true) : null;
		if ($ttLink && $link) {
			$ttLink->setDefault($link);
			/** @var array $parsedLink */
			$parsedLink = $ttLink->render(['multiLang' => $lang]);

			if ($parsedLink) {
				if (Strings::startsWith($parsedLink['link'], 'http')) {
					$text->url = $parsedLink['link'];
				} else {
					$text->url = $baseUrl . $parsedLink['link'];
				}
			}
		}

		return $text;
	}
}
