<?php declare(strict_types = 1);

namespace MultihubDropShip\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Core\Model\UI\AbstractPresenter;
use EshopOrders\AdminModule\Components\Order\OrderStatusForm;
use EshopOrders\AdminModule\Model\Event\InvoiceGenerateEvent;
use EshopOrders\AdminModule\Model\Event\InvoiceRegenerateEvent;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Pdf\IInvoicePdfBuilderFactory;
use Exception;
use MultihubDropShip\Model\Api\OrderApi;
use MultihubDropShip\Model\Clients;
use MultihubDropShip\Model\Entities\DropShipOrder;
use MultihubDropShip\Model\MultihubDropShipLogger;
use MultihubDropShip\Model\Orders;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderStatusSubscriber implements EventSubscriberInterface
{
	protected static array $processed = [];

	public function __construct(
		protected EntityManagerDecorator    $em,
		protected Orders                    $orders,
		protected OrderApi                  $orderApi,
		protected Clients                   $clients,
		protected IInvoicePdfBuilderFactory $invoicePdfBuilderFactory,
		protected Invoices                  $invoices,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			OrderStatusForm::class . '::formSuccess' => 'formSuccess',
			Order::class . '::changeStatus'          => 'orderChangeStatus',
			'eshopOrders.afterGenerateInvoice'       => 'afterGenerateInvoice',
			'eshopOrders.afterRegenerateInvoice'     => 'afterRegenerateInvoice',
		];
	}

	/**
	 * @param FormSuccessEvent $event
	 *
	 * @throws Exception
	 */
	public function formSuccess(FormSuccessEvent $event): void
	{
		/** @var OrderStatus $entity */
		$entity    = $event->custom['entity'];
		$order     = $entity->getOrder();
		$presenter = $event->presenter;
		/** @var string $value */
		$value = $event->values->status;

		$this->onOrderSetCanceled($order, $value, $presenter);
	}

	public function orderChangeStatus(OrderStatusEvent $event): void
	{
		if (!$event->order->getId() || array_key_exists($event->order->getId(), self::$processed)) {
			return;
		}

		$this->onOrderSetCanceled($event->order, $event->status, $event->presenter);
	}

	protected function onOrderSetCanceled(Order $order, string $value, ?AbstractPresenter $presenter): void
	{
		$orderId = $order->getId();

		if (!$orderId || array_key_exists($orderId, self::$processed)) {
			return;
		}

		self::$processed[$order->getId()] = $order->getId();

		if (!($dropShipOrder = $this->orders->getOrderById($orderId))) {
			return;
		}

		$clientId = $dropShipOrder->client->getId();
		if (!$clientId || !($client = $this->clients->getClient($clientId))) {
			return;
		}

		if (EshopOrdersConfig::load('invoice.enable') && !$order->getInvoice() && $order->isPaid) {
			try {
				$this->invoices->getOneByOrder($dropShipOrder->order, true);
			} catch (Exception $e) {
				MultihubDropShipLogger::log($e);
				LogNotifier::toDevelopers($e->getMessage(), 'Multihub DropShip');
			}
		}

		if ($value === OrderStatus::STATUS_PROCESSING) {
			$this->orderApi->setProcessing($client, $dropShipOrder->dropShipId);
		} else if ($value === OrderStatus::STATUS_FINISHED) {
			$this->orderApi->setCompleted($client, $dropShipOrder->dropShipId);
			$dropShipOrder->lastStatus = DropShipOrder::StatusCompleted;
			$this->em->persist($dropShipOrder);
			$this->em->flush();

			if (EshopOrdersConfig::load('invoice.enable') && !$order->getInvoice()) {
				try {
					$this->invoices->getOneByOrder($dropShipOrder->order, true);
				} catch (Exception $e) {
					MultihubDropShipLogger::log($e);
					LogNotifier::toDevelopers($e->getMessage(), 'Multihub DropShip');
				}
			}
		} else if ($value === OrderStatus::STATUS_CANCELED) {
			if ($dropShipOrder->lastStatus !== DropShipOrder::StatusCancelled) {
				$this->orderApi->setCancelled($client, $dropShipOrder->dropShipId);
				$dropShipOrder->lastStatus = DropShipOrder::StatusCancelled;
				$this->em->persist($dropShipOrder);
				$this->em->flush();

				foreach ($order->getOrderItems() as $v) {
					if ($v->getProduct()) {
						ProductSubscriber::$update[] = $v->getProductId();
					}
				}
			}
		}
	}

	public function afterGenerateInvoice(InvoiceGenerateEvent $event): void
	{
		$this->uploadInvoice($event->invoice);
	}

	public function afterRegenerateInvoice(InvoiceRegenerateEvent $event): void
	{
		$this->uploadInvoice($event->invoice);
	}

	protected function uploadInvoice(Invoice $invoice): void
	{
		$orderId = $invoice->order ? $invoice->order->getId() : null;
		if (!$orderId || !($dropShipOrder = $this->orders->getOrderById($orderId))) {
			return;
		}

		$clientId = $dropShipOrder->client->getId();
		if (!$clientId || !($client = $this->clients->getClient($clientId))) {
			return;
		}

		ob_start();
		$this->invoicePdfBuilderFactory->create($invoice)->render();
		$invoiceContent = ob_get_clean();

		if (!$invoiceContent) {
			throw new Exception('Invoice content is empty');
		}

		$this->orderApi->uploadInvoice($client, $dropShipOrder->dropShipId, $invoice->ident, $invoiceContent);
	}
}
