<?php declare(strict_types = 1);

namespace MultihubDropShip\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use EshopOrders\AdminModule\Model\Statuses;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Event\OrderPackageNumbers;
use Exception;
use MultihubDropShip\Model\Api\OrderApi;
use MultihubDropShip\Model\Clients;
use MultihubDropShip\Model\Entities\DropShipOrder;
use MultihubDropShip\Model\MultihubDropShipLogger;
use MultihubDropShip\Model\OrdersExported;
use MultihubDropShip\Model\Sync\SyncOrder;
use MultihubDropShip\Model\Sync\SyncShipment;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderSubscriber implements EventSubscriberInterface
{
	public function __construct(
		protected EntityManagerDecorator $em,
		protected OrdersExported         $ordersExported,
		protected SyncOrder              $syncOrder,
		protected Clients                $clients,
		protected Statuses               $statuses,
		protected OrderApi               $orderApi,
		protected SyncShipment           $syncShipment,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderOnSuccess'             => ['orderOnSuccess', 80],
			'eshopOrders.admin.orderOnSuccess'       => 'orderOnSuccess',
			'eshopOrders.orders.checkCompleted'      => 'checkCompleted',
			'eshopOrders.packageNumberAssigned'      => 'packageNumberAssigned',
			'MultihubDropShip.orders.checkCompleted' => 'checkCompleted',
		];
	}

	public function orderOnSuccess(OrderEvent $event): void
	{
		$order = $event->order;

		foreach ($order->getOrderItems() as $orderItem) {
			if (!$orderItem->getProduct()) {
				continue;
			}

			ProductSubscriber::$update[] = $orderItem->getProductId();
		}

		if (isset($event->formData['fromOrder']) && $event->formData['fromOrder'] instanceof Order && $event->formData['fromOrder']->getId() !== $order->getId()) {
			foreach ($event->formData['fromOrder']->getOrderItems() as $orderItem) {
				if (!$orderItem->getProduct()) {
					continue;
				}

				ProductSubscriber::$update[] = $orderItem->getProductId();
			}
		}
	}

	public function checkCompleted(Event $event): void
	{
		$orders = $this->ordersExported->getOrdersNotCompleted();

		foreach ($orders as $id => $dropShipOrder) {
			try {
				if (!$dropShipOrder->client->getId()) {
					continue;
				}

				$client = $this->clients->getClient($dropShipOrder->client->getId());
				if (!$client) {
					continue;
				}

				$detail = $this->syncOrder->getDetail($client, $dropShipOrder->dropShipId);
				if (!$detail) {
					$dropShipOrder->lastStatus = DropShipOrder::StatusCancelled;
					$this->em->persist($dropShipOrder);
					$this->em->flush();

					MultihubDropShipLogger::log('Check completed - Detail ' . $id . ' not found - ' . $client->getName() . ' - dropship id: ' . $dropShipOrder->dropShipId);

					continue;
				}

				if ($detail['status'] === DropShipOrder::StatusCompleted) {
					$this->statuses->changeStatus([$dropShipOrder->order->getId()], OrderStatus::STATUS_FINISHED);
				}
			} catch (Exception $e) {
				MultihubDropShipLogger::log($e);
				LogNotifier::toDevelopers($e->getMessage(), 'Multihub DropShip');
			}
		}
	}

	public function packageNumberAssigned(OrderPackageNumbers $event): void
	{
		if (!$event->packageNumbers) {
			return;
		}

		/** @var DropShipOrder|null $order */
		$order = $this->em->getRepository(DropShipOrder::class)->findOneBy(['order' => $event->order->getId()]);

		if ($order) {
			$client = $this->clients->getClient($order->client->getId());
			if (!$client) {
				return;
			}

			$r = $this->orderApi->addTrackingCode($client, $order->dropShipId, $event->packageNumbers[0]);
			if ($r) {
				$this->syncShipment->setShipped($client, $order->dropShipId);
				$order->setExported();
				$order->trackingNumber = $event->packageNumbers[0];
				$order->trackingUrl    = $event->trackingUrls[0] ?? '';

				$this->em->persist($order);
				$this->em->flush();
			}
		}
	}
}
