<?php declare(strict_types = 1);

namespace MultihubDropShip\Console;

use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use Doctrine\ORM\Tools\SchemaTool;
use MultihubDropShip\Model\Api\AccountApi;
use MultihubDropShip\Model\CacheService;
use MultihubDropShip\Model\Clients;
use Nette\Utils\Json;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class RatingDownload extends AbstractCommand
{
	protected CacheService           $cacheService;
	protected EntityManagerDecorator $em;
	protected AccountApi             $accountApi;
	protected Clients                $clients;

	public function __construct(
		CacheService           $cacheService,
		EntityManagerDecorator $em,
		AccountApi             $accountApi,
		Clients                $clients,
		?string                $name = null
	)
	{
		parent::__construct($name);
		$this->cacheService = $cacheService;
		$this->em           = $em;
		$this->accountApi   = $accountApi;
		$this->clients      = $clients;
	}

	protected function configure(): void
	{
		$this->setName('multihub-drop-ship:rating-download');
	}

	protected function executeSchemaCommand(InputInterface $input, OutputInterface $output, SchemaTool $schemaTool, array $metadatas, SymfonyStyle $ui)
	{
		$output->writeln('Multihub rating download');

		$conn = $this->em->getConnection();
		try {
			foreach ($this->clients->getClients() as $client) {
				$data = $this->accountApi->getAccount($client);

				if (isset($data['rating'])) {
					if (isset($data['rating']['allegro'])) {
						$conn->update('multihub_drop_ship__client', [
							'allegro_rating' => Json::encode($data['rating']['allegro']),
						], [
							'id' => $client->getId(),
						]);

						$this->cacheService->clearRating($client->getId());
					}
				}
			}

			$output->writeLn('<info>[OK] - downloaded</info>');

			return 0;
		} catch (\Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 1;
	}

}
