<?php declare(strict_types = 1);

namespace MultihubDropShip\Model;

use Core\Model\Countries;
use Core\Model\Entities\Country;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Strings;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Currency\Model\Currencies;
use Currency\Model\Entities\Currency;
use Currency\Model\Exchange;
use DateTime;
use EshopCatalog\AdminModule\Model\Sellers;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\AdminModule\Model\Payments;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\AdminModule\Model\Statuses;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Entities\OrderCurrency;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\Entities\Spedition;
use MultihubDropShip\Model\Dao\Client;
use MultihubDropShip\Model\Entities\DropShipOrder;
use MultihubDropShip\Model\Subscribers\ProductSubscriber;
use Nette\DI\Container;

class Orders
{
	protected Container              $container;
	protected EntityManagerDecorator $em;
	protected Sellers                $sellers;
	protected Speditions             $speditions;
	protected Payments               $payments;
	protected Currencies             $currencies;
	protected Exchange               $exchange;
	protected Statuses               $statusesService;
	protected Countries              $countries;
	protected EventDispatcher        $eventDispatcher;

	public function __construct(
		Container              $container,
		EntityManagerDecorator $em,
		Sellers                $sellers,
		Speditions             $speditions,
		Currencies             $currencies,
		Exchange               $exchange,
		Payments               $payments,
		Statuses               $statusesService,
		Countries              $countries,
		EventDispatcher        $eventDispatcher
	)
	{
		$this->container  = $container;
		$this->em         = $em;
		$this->sellers    = $sellers;
		$this->speditions = $speditions;
		$this->currencies = $currencies;
		$this->exchange   = $exchange;
		$this->payments   = $payments;

		$this->statusesService = $statusesService;
		$this->countries       = $countries;
		$this->eventDispatcher = $eventDispatcher;
	}

	public function addOrder(Client $client, array $data): ?DropShipOrder
	{
		if ($this->getOrderByDropShipId($data['id'])) {
			return null;
		}

		$productsForUpdate = [];

		$seller = null;
		$this->em->beginTransaction();
		try {
			if ($data['marketplace'] === 'allegro') {
				$seller = $this->sellers->get($client->sellerId);
			} else if ($data['marketplace'] === 'kaufland') {
				$seller = $this->sellers->get($client->sellerIdKaufland);
			}

			/** @var \MultihubDropShip\Model\Entities\Client $clientEntity */
			$clientEntity = $this->em->getRepository(\MultihubDropShip\Model\Entities\Client::class)->find($client->getId());

			if (!$seller || $seller->getSites()->isEmpty()) {
				throw new \Exception("Seller {$client->sellerId} not found");
			}

			$site = $seller->getSites()->first()->getSite();

			/** @var Spedition|null $eshopSpedition */
			$eshopSpedition = null;
			if (isset($data['delivery']['externalId'])) {
				$eshopSpedition = $this->speditions->getEr()->findOneBy(['externalIdent' => $data['delivery']['externalId']]);
			}

			if (!$eshopSpedition) {
				if ($data['marketplace'] === 'allegro') {
					$eshopSpedition = $this->speditions->getEr()->findOneBy(['ident' => 'allegro']);
				}
			}

			if ($eshopSpedition instanceof Spedition === false) {
				throw new \Exception("Spedition allegro not found");
			}

			if ($data['payment']['ident'] === 'ONLINE') {
				$paymentIdent = 'card';
			} else if ($data['payment']['ident'] === 'CASH_ON_DELIVERY' || $data['payment']['isCashOnDelivery']) {
				$paymentIdent = 'cod';
			} else if ($data['marketplace'] === 'kaufland') {
				$paymentIdent = 'kaufland';
			} else {
				$paymentIdent = null;
			}

			/** @var Payment|null $eshopPayment */
			$eshopPayment = $paymentIdent ? $this->payments->getEr()->findOneBy(['ident' => $paymentIdent]) : null;

			$order = new Order($site);
			$order->setAgreedTerms(true);

			// TODO from invoice required
			$invData = $data['invoiceAddress'];
			$addrInv = new OrderAddress(OrderAddress::ADDRESS_INVOICE);
			$addrInv->setFirstName($invData['firstName'] ?: '');
			$addrInv->setLastName($invData['lastName'] ?: '');
			$addrInv->setPhone($invData['phone'] ?: '');
			$addrInv->setEmail($invData['email']);
			$addrInv->setStreet($invData['street']);
			$addrInv->setCity($invData['city']);
			$addrInv->setPostal($invData['zipCode']);
			$addrInv->setCompany($invData['companyName']);
			$addrInv->setIdNumber($invData['idNumber']);
			$addrInv->setVatNumber($invData['vatNumber']);

			$countryEntity = null;
			if ($invData['countryCode']) {
				/** @var ?Country $countryEntity */
				$countryEntity = $this->em->getRepository(Country::class)->findOneBy(['id' => [
					Strings::lower($invData['countryCode']),
					Strings::upper($invData['countryCode']),
				]]);
			}

			if (!$countryEntity) {
				throw new \Exception("Country {$invData['countryCode']} not found");
			}

			$addrInv->setCountry($countryEntity);

			$this->em->persist($addrInv);
			$order->setAddressInvoice($addrInv);

			switch ($invData['countryCode']) {
				case 'PL':
					$order->lang = 'pl';
					break;
				case'SK':
					$order->lang = 'sk';
					break;
				case 'DE':
					$order->lang = 'de';
					break;
				default:
					$order->lang = 'cs';
			}

			$deliData = $data['deliveryAddress'];
			$addrDeli = new OrderAddress(OrderAddress::ADDRESS_DELIVERY);
			$addrDeli->setFirstName($deliData['firstName'] ?: '');
			$addrDeli->setLastName($deliData['lastName'] ?: '');
			$addrDeli->setPhone($deliData['phone'] ?: '');
			$addrDeli->setEmail($deliData['email']);
			$addrDeli->setStreet($deliData['street']);
			$addrDeli->setCity($deliData['city']);
			$addrDeli->setPostal($deliData['zipCode']);

			$deliCountryEntity = null;
			if ($deliData['countryCode']) {
				/** @var ?Country $deliCountryEntity */
				$deliCountryEntity = $this->em->getRepository(Country::class)->findOneBy(['id' => [
					Strings::lower($deliData['countryCode']),
					Strings::upper($deliData['countryCode']),
				]]);
			}

			if (!$deliCountryEntity) {
				throw new \Exception("Country {$deliData['countryCode']} not found");
			}

			$addrDeli->setCountry($deliCountryEntity);

			$this->em->persist($addrDeli);
			$order->setAddressDelivery($addrDeli);

			$items          = [];
			$currencyRate   = null;
			$productVatRate = null;

			/** @var Currency|null $currency */
			$currency         = null;
			$currencyDecimals = 2;

			if ($data['currency'] !== $this->currencies->getDefaultCode()) {
				/** @var Currency|null $currency */
				$currency = $this->currencies->getBySite('allegro')[$data['currency']];
				if (!$currency) {
					throw new \Exception("Currency {$data['currency']} not found");
				}

				$currencyDecimals = $currency->decimals;
			}

			$itemsSorted = [];
			foreach ($data['items'] as $k => $item) {
				/** @var Product|null $prod */
				$prod = $item['remoteId'] ? $this->em->getRepository(Product::class)->find($item['remoteId']) : null;

				$item['product']          = $prod;
				$item['manufacturerName'] = $prod && $prod->getManufacturer() ? $prod->getManufacturer()->name : null;
				$item['productCode1']     = $prod->code1 ?? null;

				$itemsSorted[$k] = $item;
			}

			usort($itemsSorted, static function($a, $b) {
				if ($a['manufacturerName'] && $b['manufacturerName']
					&& $a['manufacturerName'] !== $b['manufacturerName']) {
					return $a['manufacturerName'] <=> $b['manufacturerName'];
				}

				if ($a['productCode1'] && $b['productCode1']
					&& $a['productCode1'] !== $b['productCode1']) {
					return $a['productCode1'] <=> $b['productCode1'];
				}

				return 1;
			});


			foreach ($itemsSorted as $item) {
				/** @var Product|null $prod */
				$prod = $item['product'];

				$productVatRate = $item['vatRate'] ? (int) $item['vatRate'] : 21;

				$orderItem = new OrderItem($prod, $order);
				$orderItem->setQuantity((int) $item['quantity']);
				$orderItem->setVatRate($productVatRate);

				$code1      = $item['code1'];
				$namePrefix = '';

				if ($prod) {
					if ($prod->getManufacturer()) {
						$namePrefix = $prod->getManufacturer()->name . ' ';
					}

					if (!$code1) {
						$code1 = $prod->code1;
					}
				}

				if ($item['remoteId']) {
					$productsForUpdate[] = $item['remoteId'];
				}

				$orderItem->setCode1($code1);

				$price = $item['price'];

				if ($currency) {
					if (!$currencyRate) {
						if ($prod) {
							$currencyRate = round($prod->price / $price, 5);
						} else {
							$price = round($price * (float) $currency->rate, $currencyDecimals);
						}
					}

					if ($currencyRate) {
						$price = round($price * $currencyRate, $currencyDecimals);
					}
				}
				$orderItem->setPrice($price);

				$orderItemText = $orderItem->addOrderItemText('cs');
				$orderItemText->setName($namePrefix . $item['name']);

				$this->em->persist($orderItem);
				$items[] = $orderItem;
			}

			$order->setOrderItems($items);

			if ($currency && $currencyRate) {
				$orderCurrency           = new OrderCurrency($order, $data['currency']);
				$orderCurrency->rate     = $currencyRate;
				$orderCurrency->decimals = $currencyDecimals;

				$order->currency = $orderCurrency;
			}

			$orderSpedition = new OrderSpedition($eshopSpedition, $order);
			$price          = $data['delivery']['price'];
			if ($currency && $currencyRate) {
				$price = round($price * $currencyRate, $currencyDecimals);
			}

			if ($productVatRate) {
				$orderSpedition->setVatRate($productVatRate);
			} else if ($eshopSpedition->vatRate) {
				$orderSpedition->setVatRate($eshopSpedition->vatRate->rate);
			}

			$orderSpedition->setPrice($price);
			$order->setSpedition($orderSpedition);
			$this->em->persist($orderSpedition);

			$orderPayment = null;
			if ($eshopPayment) {
				$orderPayment = new OrderPayment($eshopPayment, $order);
				$price        = $data['payment']['price'];
				if ($currency && $currencyRate) {
					$price = round((float) $price * $currencyRate, $currencyDecimals);
				}
				$orderPayment->setPrice($price ?: 0);
				$this->em->persist($orderPayment);
				$order->setPayment($orderPayment);
			}

			$order->isPaid = (int) $data['isPaid'];

			if ($data['note']) {
				$order->setMessage((string) $data['note']);
			}

			$statusCreated = $this->statusesService->get('created');
			if (!$statusCreated) {
				throw new \Exception("Status 'created' not found");
			}

			$orderActionCreated = new OrderStatus($order, $statusCreated);
			$order->getOrderStatuses()->add($orderActionCreated);

			$eventOrder                 = new OrderEvent($order);
			$eventOrder->addrDelivery   = $addrDeli;
			$eventOrder->addrInvoice    = $addrInv;
			$eventOrder->orderSpedition = $orderSpedition;
			$eventOrder->orderPayment   = $orderPayment;
			$eventOrder->orderItems     = $items;
			$this->eventDispatcher->dispatch($eventOrder, 'eshopOrders.orderBeforeSave');

			$this->em->persist($order);
			$this->em->persist($orderSpedition);

			if ($orderPayment) {
				$this->em->persist($orderPayment);
			}

			$dropShipOrder              = new DropShipOrder(
				$clientEntity,
				$order,
				$data['id']
			);
			$dropShipOrder->marketplace = $data['marketplace'];
			$this->em->persist($dropShipOrder);
			$this->em->flush();

			$this->em->commit();

			foreach ($productsForUpdate as $v) {
				$this->em->getConnection()->executeQuery("INSERT IGNORE INTO multihub_drop_ship__product_update_list (`product_id`, `created`) VALUES (:id, :created)", [
					'id'      => $v,
					'created' => (new DateTime())->format('Y-m-d H:i:s'),
				]);
			}

			return $dropShipOrder;
		} catch (\Exception $e) {
			MultihubDropShipLogger::log($e);
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}

			LogNotifier::toDevelopers($e->getMessage() . ' ' . $e->getTraceAsString());
		}

		return null;
	}

	public function getOrderByDropShipId(string $id): ?DropShipOrder
	{
		/** @var DropShipOrder|null $object */
		$object = $this->em->getRepository(DropShipOrder::class)->findOneBy(['dropShipId' => $id]);

		return $object;
	}

	public function getOrderById(int $id): ?DropShipOrder
	{
		/** @var DropShipOrder|null $object */
		$object = $this->em->getRepository(DropShipOrder::class)->findOneBy(['order' => $id]);

		return $object;
	}

	public function setCancelled(DropShipOrder $dropShipOrder): void
	{
		$statusCancelled = $this->statusesService->get('canceled');
		if ($statusCancelled) {
			$orderActionCancelled = new OrderStatus($dropShipOrder->order, $statusCancelled);

			$this->em->persist($orderActionCancelled);

			$dropShipOrder->lastStatus = DropShipOrder::StatusCancelled;
			$this->em->persist($dropShipOrder);
			$this->em->flush();
		}
	}
}
