<?php declare(strict_types = 1);

namespace MultihubDropShip\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Arrays;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\Model\Entities\OrderStatus;
use MultihubDropShip\Model\Entities\DropShipOrder;

class OrdersExported
{
	protected EntityManagerDecorator $em;

	public function __construct(
		EntityManagerDecorator $em
	)
	{
		$this->em = $em;
	}

	/**
	 * @param array $ids
	 *
	 * @return DropShipOrder[]
	 */
	public function getOrdersNotExported(array $ids = []): array
	{
		$orders = [];
		foreach ($this->em->getRepository(DropShipOrder::class)->createQueryBuilder('dso')
			         ->addSelect('o, oCurrency')
			         ->innerJoin('dso.order', 'o', Join::WITH, 'o.id IN (:ids)')
			         ->leftJoin('o.currency', 'oCurrency')
			         ->where('dso.exported IS NULL')
			         ->setParameter('ids', $ids)
			         ->getQuery()->getResult() as $dso) {
			/** @var DropShipOrder $dso */
			$orders[$dso->order->getId()] = $dso;
		}

		$used = [];
		foreach ($this->em->createQueryBuilder()->select('IDENTITY(os.order) as order, IDENTITY(os.status) as status')
			         ->from(OrderStatus::class, 'os')
			         ->where('os.order IN (:ids)')
			         ->setParameters([
				         'ids' => array_keys($orders),
			         ])->orderBy('os.created', 'DESC')
			         ->getQuery()->getArrayResult() as $row) {
			if (Arrays::contains($used, $row['order'])) {
				continue;
			}

			$used[] = $row['order'];

			if ($row['status'] !== OrderStatus::STATUS_PROCESSING) {
				unset($orders[$row['order']]);
			}
		}

		return $orders;
	}

	/**
	 * @param array $ids
	 *
	 * @return DropShipOrder[]
	 */
	public function getByIds(array $ids): array
	{
		$arr = [];
		foreach ($this->em->getRepository(DropShipOrder::class)->createQueryBuilder('dso')
			         ->where('dso.order IN (:ids)')
			         ->setParameter('ids', $ids)
			         ->getQuery()->getResult() as $row) {
			/** @var DropShipOrder $row */
			$arr[$row->order->getId()] = $row;
		}

		return $arr;
	}

	/**
	 * @param array $ids
	 *
	 * @return DropShipOrder[]
	 */
	public function getExported(array $ids): array
	{
		$arr = [];
		foreach ($this->em->getRepository(DropShipOrder::class)->createQueryBuilder('dso')
			         ->where('dso.order IN (:ids)')
			         ->andWhere('dso.exported IS NOT NULL')
			         ->innerJoin('dso.order', 'o')
			         ->setParameter('ids', $ids)
			         ->getQuery()->getResult() as $row) {
			/** @var DropShipOrder $row */
			$arr[$row->order->getId()] = $row;
		}

		return $arr;
	}

	//	public function getOrders(array $ids): array
	//	{
	//		$data = [];
	//
	//		foreach ($this->em->createQueryBuilder()
	//			         ->select('IDENTITY(ao.order) as order, ao.alzaId, ao.exported, ao.shippingCarrierCode')
	//			         ->from(AlzaOrder::class, 'ao')
	//			         ->where('ao.order IN (:ids)')
	//			         ->setParameter('ids', $ids)
	//			         ->getQuery()->getArrayResult() as $row) {
	//			$data[$row['order']] = $row;
	//		}
	//
	//		return $data;
	//	}

	public function findIdByPackageNumber(string $packageNumber): array
	{
		$ids = [];

		foreach ($this->em->getRepository(DropShipOrder::class)->createQueryBuilder('dso')
			         ->select('IDENTITY(dso.order) as id')
			         ->where('dso.trackingNumber LIKE :number')
			         ->setParameter('number', "%$packageNumber%")
			         ->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		return $ids;
	}

	public function findIdByDropShipId(string $dropShipId): array
	{
		$ids = [];
		foreach ($this->em->getRepository(DropShipOrder::class)->createQueryBuilder('dso')
			         ->select('IDENTITY(dso.order) as id')
			         ->where('dso.dropShipId LIKE :dropShipId')
			         ->setParameter('dropShipId', "%$dropShipId%")
			         ->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		return $ids;
	}

	/**
	 * @return DropShipOrder[]
	 */
	public function getOrdersNotCompleted(): array
	{
		$data = [];

		foreach ($this->em->createQueryBuilder()->select('dso')
			         ->from(DropShipOrder::class, 'dso')
			         ->where('dso.lastStatus NOT IN (:status)')
			         ->setParameters([
				         'status' => [
					         DropShipOrder::StatusCompleted,
					         DropShipOrder::StatusCancelled,
				         ],
			         ])->getQuery()->getResult() as $row) {
			/** @var DropShipOrder $row */
			$data[$row->order->getId()] = $row;
		}

		return $data;
	}

	//	public function setLastStatus(array $orderIds, string $status = MallOrder::STATUS_DELIVERED): void
	//	{
	//		$this->em->getConnection()->executeStatement("UPDATE alza__order SET last_status = ? WHERE order_id IN (" . implode(',', $orderIds) . ")", [
	//			$status,
	//		]);
	//	}
}
