<?php declare(strict_types = 1);

namespace MultihubDropShip\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Helpers\Arrays;
use EshopOrders\AdminModule\Model\Dao\OrderExpedition;
use EshopOrders\AdminModule\Model\Orders as EshopOrders;
use EshopOrders\Model\Entities\OrderHistory;
use MultihubDropShip\Model\Clients;
use MultihubDropShip\Model\Entities\DropShipOrder;
use MultihubDropShip\Model\OrdersExported;
use MultihubDropShip\Model\Sync\SyncShipment;
use Nette\Utils\FileSystem;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class ExpeditionSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected OrdersExported         $ordersExported;
	protected SyncShipment           $syncShipment;
	protected Clients                $clients;

	public function __construct(
		EntityManagerDecorator $em,
		OrdersExported         $ordersExported,
		SyncShipment           $syncShipment,
		Clients                $clients
	)
	{
		$this->em             = $em;
		$this->ordersExported = $ordersExported;
		$this->syncShipment   = $syncShipment;
		$this->clients        = $clients;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			EshopOrders::class . '::processOrdersForExpedition' => 'processOrdersForExpedition',
			EshopOrders::class . '::sendExpeditionData'         => ['sendExpeditionData', 100],
			EshopOrders::class . '::generateLabel'              => 'generateLabel',
			EshopOrders::class . '::afterGenerateLabel'         => 'afterGenerateLabel',
		];
	}

	public function processOrdersForExpedition(Event $event): void
	{
		$orderExpeditions = [];
		$ids              = $this->filterOrders($event->data['data']);

		if ($ids) {
			foreach ($this->ordersExported->getOrdersNotExported($ids) as $id => $mo) {
				$orderExpeditions[$id] = new OrderExpedition($mo->order, (string) $mo->order->getSpeditionIdent(), $mo);
			}

			$event->data['expedition'] += $orderExpeditions;
		}
	}

	public function sendExpeditionData(Event $event): void
	{
		$ids = $this->filterOrders($event->data['data']);

		if ($ids) {
			$orders   = $this->ordersExported->getOrdersNotExported($ids);
			$quantity = $event->data['quantity'] ?? 1;

			foreach ($orders as $dropShipOrder) {
				$client = $this->clients->getClient($dropShipOrder->client->getId());
				if (!$client) {
					$result = false;
				} else {
					$result = $this->syncShipment->setPacking($client, $dropShipOrder->dropShipId, (int) $quantity);
				}

				if ($result) {
					$event->data['okCount']++;
					$dropShipOrder->lastStatus = DropShipOrder::StatusPacking;
					$this->em->persist($dropShipOrder);
					$this->em->flush($dropShipOrder);
				} else {
					$event->data['errorCount']--;
				}
			}
		}
	}

	public function generateLabel(Event $event): void
	{
		$ids = $this->filterOrders($event->data['data']);

		if ($ids) {
			foreach ($this->ordersExported->getByIds($ids) as $dropShipOrder) {
				$client = $this->clients->getClient($dropShipOrder->client->getId());

				if (!$client) {
					$event->data['errorCount']++;

					$orderHistory = new OrderHistory(
						$dropShipOrder->order,
						'DropShip ERROR - klient nenalezen'
					);
					$this->em->persist($orderHistory);

					continue;
				}

				$result = $this->syncShipment->getLabel($client, $dropShipOrder->dropShipId);
				if ($result) {
					FileSystem::createDir(TMP_DIR . '/dropShipLabels');

					if (file_put_contents(TMP_DIR . '/dropShipLabels/stitky.pdf', base64_decode($result))) {
						$event->data['files'][] = TMP_DIR . '/dropShipLabels/stitky.pdf';
						$event->data['okCount']++;
						$orderMsg = 'DropShip OK - Vygenerování štítku';
					} else {
						$event->data['errorCount']++;
						$orderMsg = 'DropShip ERROR - Vygenerování štítku';
					}
				} else {
					$event->data['errorCount']++;
					$orderMsg = 'DropShip ERROR - Vygenerování štítku';
				}

				$orderHistory = new OrderHistory(
					$dropShipOrder->order,
					$orderMsg,
				);
				$this->em->persist($orderHistory);
			}
		}
	}

	public function afterGenerateLabel(Event $event): void
	{
		$ids = $this->filterOrders($event->data['data']);

		if ($ids) {
			foreach ($this->ordersExported->getByIds($ids) as $dropShipOrder) {
				$client = $this->clients->getClient($dropShipOrder->client->getId());

				if (!$client) {
					$event->data['errorCount']--;
					continue;
				}

				$result = $this->syncShipment->setShipped($client, $dropShipOrder->dropShipId);

				if ($result['order'] ?? null) {
					$event->data['okCount']++;

					$dropShipOrder->setExported();

					if (!empty($result['trackingNumbers'])) {
						$tmp                           = array_values($result['trackingNumbers'])[0];
						$dropShipOrder->trackingNumber = $tmp['id'];
						$dropShipOrder->trackingUrl    = $tmp['url'];
					}

					$this->em->persist($dropShipOrder);
					$this->em->flush($dropShipOrder);
				} else {
					$event->data['errorCount']--;
				}
			}
		}
	}

	/**
	 * @param array $data
	 *
	 * @return array
	 */
	protected function filterOrders(array $data): array
	{
		$ids = [];
		foreach ($data as $v) {
			if (Arrays::contains(['allegro', 'allegroPickup'], (string) $v['service'])) {
				$ids[] = $v['id'];
			}
		}

		return $ids;
	}
}
