<?php declare(strict_types = 1);

namespace MultihubDropShip\Model\Sync;

use Core\Model\Sites;
use MultihubDropShip\Model\Api\ProductsApi;
use MultihubDropShip\Model\Clients;
use MultihubDropShip\Model\EshopProducts;
use MultihubDropShip\Model\ProductBuilder;

class SyncProducts
{
	protected Clients        $clients;
	protected EshopProducts  $eshopProducts;
	protected ProductBuilder $productBuilder;
	protected ProductsApi    $productsApi;

	public function __construct(
		Clients        $clients,
		EshopProducts  $eshopProducts,
		ProductBuilder $productBuilder,
		ProductsApi    $productsApi
	)
	{
		$this->clients        = $clients;
		$this->eshopProducts  = $eshopProducts;
		$this->productBuilder = $productBuilder;
		$this->productsApi    = $productsApi;
	}

	public function memoryUsage(): string
	{
		$mem_usage = memory_get_usage(true);
		if ($mem_usage < 1024) {
			$mem_usage .= ' bytes';
		} else if ($mem_usage < 1048576) {
			$mem_usage = round($mem_usage / 1024, 2) . ' kilobytes';
		} else {
			$mem_usage = round($mem_usage / 1048576, 2) . ' megabytes';
		}

		return $mem_usage;
	}

	public function updateAll(int $clientId): void
	{
		$client = $this->clients->getClient($clientId);
		if (!$client) {
			return;
		}

		Sites::$currentIdentOverride = $client->siteIdent;
		Sites::$currentLangOverride  = $client->lang;

		$productsIds = $this->eshopProducts->findProductsIdsByCategories($client);
		if (empty($productsIds)) {
			return;
		}

		$hubProducts = $this->productsApi->getProducts($client);

		foreach (array_diff_key($hubProducts, array_flip($productsIds)) as $id => $hubProduct) {
			if ($hubProduct['status']['name'] !== 'Disabled') {
				$result = $this->productsApi->updateStatus($client, (string) $id, 'disabled');
			}
			// todo result false pridat log a notifikaci?
		}

		foreach (array_chunk($productsIds, 500) as $chunk) {
			foreach ($this->productBuilder->buildProducts($client, $chunk) as $product) {
				$this->productsApi->updateProduct($client, $product);
			}
		}
	}

	public function sendAllTo(int $clientId, string $service): void
	{
		$client = $this->clients->getClient($clientId);
		if (!$client) {
			return;
		}

		Sites::$currentIdentOverride = $client->siteIdent;
		Sites::$currentLangOverride  = $client->lang;

		$productsIds = $this->eshopProducts->findProductsIdsByCategories($client);
		if (empty($productsIds)) {
			return;
		}

		$hubProducts = $this->productsApi->getProducts($client);

		foreach (array_intersect_key($hubProducts, array_flip($productsIds)) as $id => $hubProduct) {
			$this->productsApi->sendTo($client, (string) $id, 'allegro');
		}
	}

}
