<?php declare(strict_types = 1);

namespace MultihubDropShip\Console;

use Core\Model\Application\AppState;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use Doctrine;
use EshopCatalog\AdminModule\Model\Categories;
use Exception;
use MultihubDropShip\Model\Api\CategoriesApi;
use MultihubDropShip\Model\CategoryBuilder;
use MultihubDropShip\Model\Clients;
use MultihubDropShip\Model\EshopCategories;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Tracy\Debugger;

#[AsCommand(name: 'multihub-drop-ship:category-update-from-list')]
class CategoryUpdateFromList extends Command
{
	public function __construct(
		protected CategoriesApi          $categoriesApi,
		protected EntityManagerDecorator $em,
		protected Clients                $clients,
		protected EshopCategories        $eshopCategories,
		protected CategoryBuilder        $categoryBuilder,
		protected Categories             $adminCategories,
	)
	{
		parent::__construct();
	}

	protected function configure(): void
	{
		$this->setName('multihub-drop-ship:category-update-from-list')
			->setDescription('Update from db list');
	}

	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		$tmpFile = TMP_DIR . '/multihubCategoryUpdating.txt';

		$lastRun = file_exists($tmpFile) ? (int) file_get_contents($tmpFile) : null;
		$now     = time();

		if ($lastRun && ($now - $lastRun) < 1300) {
			$output->writeln('Multihub DropShip - category update - already running');

			return 0;
		}

		$output->writeln('Multihub DropShip - category update');

		try {
			Debugger::dispatch();
			$conn = $this->em->getConnection();

			file_put_contents($tmpFile, time());

			try {
				$allIds = [];
				foreach ($conn->fetchAllAssociative("SELECT category_id FROM multihub_drop_ship__category_update_list") as $row) {
					/** @var array $row */
					$allIds[] = (string) $row['category_id'];
				}

				if (!empty($allIds)) {
					foreach ($this->clients->getClients() as $client) {
						Sites::$currentIdentOverride = $client->siteIdent;
						Sites::$currentLangOverride  = $client->lang;

						// Vyfiltrovani kategorii podle siteIdent
						$ids = array_intersect($allIds, $this->eshopCategories->getCategories($client));

						if (!empty($ids)) {
							AppState::setState('eshopCatalogLinkTest', [
								'source'    => 'multihub-drop-ship',
								'siteIdent' => $client->siteIdent,
								'lang'      => $client->lang,
								'ids'       => $ids,
							]);

							$categories = $this->categoryBuilder->buildCategories($client, $ids);

							try {
								// Aktualizace
								foreach ($categories as $id => $category) {
									$res = $this->categoriesApi->updateCategory($client, $category);

									if (isset($res['accepted']) || isset($res['categoryId'])) {
									} else {
										Debugger::log($res, 'multihub-drop-ship-category-update');
									}
								}
							} catch (Exception $e) {
							}
						}
					}

					// Odebrani z db
					$conn->executeQuery("DELETE FROM multihub_drop_ship__category_update_list WHERE `category_id` IN ('" . implode("','", $allIds) . "')");
				}

				$output->writeLn('<info>[OK] - categories updated</info>');
			} catch (Exception $e) {
				$output->writeLn('<info>[ERR] - failed - ' . $e->getMessage() . '</info>');
				Debugger::log($e, 'multihub-drop-ship-category-update');
			}

			unlink($tmpFile);

			return 0;
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 1;
	}
}
