<?php declare(strict_types = 1);

namespace MultihubDropShip\Console;

use Doctrine;
use Exception;
use MultihubDropShip\Model\Clients;
use MultihubDropShip\Model\Dao\Client;
use MultihubDropShip\Model\Sync\SyncOrder;
use Override;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Tracy\Debugger;

#[AsCommand(name: 'multihub-drop-ship:download-orders')]
class DownloadOrders extends Command
{
	public function __construct(
		protected SyncOrder $syncOrders,
		protected Clients   $clients,
	)
	{
		parent::__construct();
	}

	#[Override]
	protected function configure(): void
	{
		$this->setName('multihub-drop-ship:download-orders')
			->addOption('clientId', null, InputOption::VALUE_REQUIRED)
			->setDescription('Download new orders');
	}

	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		$output->writeln('<fg=white;bg=#007bff>Multihub - Download orders</>');
		try {
			/** @var int $clientId */
			$clientId = $input->getOption('clientId');

			$client = $this->clients->getClient((int) $clientId);
			if ($client instanceof Client) {
				$this->syncOrders->downloadNew($client);
				$output->writeLn('<info>[OK] - orders downloaded</info>');

				try {
					$this->syncOrders->downloadCancelled($client);
				} catch (Exception $e) {
					Debugger::log($e, 'setcancelled');
				}
			} else {
				throw new Exception('Client not found');
			}
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 1;
	}
}
