<?php declare(strict_types = 1);

namespace MultihubDropShip\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\ProductInSite;
use MultihubDropShip\Model\Dao\Client;

class EshopProducts
{
	protected array $cProducts = [];

	public function __construct(
		protected EntityManagerDecorator $em,
		protected CacheService           $cacheService,
	)
	{
	}

	public function findProductsIdsByCategories(Client $client): array
	{
		$conn = $this->em->getConnection();

		if ($client->sendAll) {
			if (!array_key_exists('all', $this->cProducts)) {
				$this->cProducts['all'] = [];

				foreach ($conn->executeQuery("SELECT product_id FROM eshop_catalog__product_in_site
						WHERE site = '{$client->siteIdent}' AND category_id IS NOT NULL")->iterateAssociative() as $row) {
					/** @var array $row */
					$this->cProducts['all'][] = (int) $row['product_id'];
				}
			}

			return $this->cProducts['all'];
		}

		$allowedCategories = $client->allowedCategoriesIds;
		$key               = md5(serialize($allowedCategories));
		if (!array_key_exists($key, $this->cProducts)) {
			$this->cProducts[$key] = [];

			foreach (array_chunk($allowedCategories, 250) as $chunk) {
				foreach ($conn->executeQuery("SELECT product_id FROM eshop_catalog__product_in_site
						WHERE site = '{$client->siteIdent}' AND category_id IN (" . implode(',', $chunk) . ")")->iterateAssociative() as $row) {
					/** @var array $row */
					$this->cProducts[$key][] = $row['product_id'];
				}

				foreach ($conn->executeQuery("SELECT id_product FROM eshop_catalog__category_product
						WHERE id_category IN (" . implode(',', $chunk) . ")")->iterateAssociative() as $row) {
					/** @var array $row */
					$this->cProducts[$key][] = $row['id_product'];
				}
			}

			$this->cProducts[$key] = array_unique($this->cProducts[$key]);
		}

		return $this->cProducts[$key];
	}

	public function isProductAllowed(int $productId, Client $client): bool
	{
		if ($client->sendAll) {
			return (bool) (((array) $this->em->getRepository(ProductInSite::class)
				->createQueryBuilder('pis')
				->select('IDENTITY(pis.product) as id')
				->where('pis.site = :site')
				->andWhere('pis.product = :productId')
				->andWhere('pis.category IS NOT NULL')
				->setParameters(new ArrayCollection([new Parameter('site', $client->siteIdent), new Parameter('productId', $productId)]))
				->getQuery()->getOneOrNullResult())['id'] ?? null);
		}

		$allowedCategories = $client->allowedCategoriesIds;
		$result            = ((array) $this->em->getRepository(ProductInSite::class)
			->createQueryBuilder('pis')
			->select('IDENTITY(pis.product) as id')
			->where('pis.category IN (:ids)')
			->andWhere('pis.product = :productId')
			->andWhere('pis.category IS NOT NULL')
			->setParameters(new ArrayCollection([new Parameter('ids', $allowedCategories), new Parameter('productId', $productId)]))
			->getQuery()->getOneOrNullResult())['id'] ?? null;

		if (!$result) {
			$result = $this->em->getRepository(CategoryProduct::class)
				->createQueryBuilder('cp')
				->select('IDENTITY(cp.product) as id')
				->andWhere('cp.category IN (:ids)')
				->andWhere('cp.product = :productId')
				->setParameters(new ArrayCollection([new Parameter('ids', $allowedCategories), new Parameter('productId', $productId)]))
				->getQuery()->getArrayResult()['id'] ?? null;
		}

		return (bool) $result;
	}
}
