<?php declare(strict_types = 1);

namespace MultihubDropShip\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\Common\EventSubscriber;
use Doctrine\ORM\Event\OnFlushEventArgs;
use Doctrine\Persistence\Event\LifecycleEventArgs;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopOrders\Model\Entities\OrderItem;
use Exception;
use MultihubDropShip\Model\Api\ProductsApi;
use MultihubDropShip\Model\Clients;
use MultihubDropShip\Model\EshopProducts;
use Override;

class ProductSubscriber implements EventSubscriber
{
	public static array $update = [];

	public function __construct(
		protected ProductsApi            $productsApi,
		protected Clients                $clients,
		protected EshopProducts          $eshopProducts,
		protected EntityManagerDecorator $em,
	)
	{
	}

	#[Override]
	public function getSubscribedEvents(): array
	{
		return [
			'postPersist',
			'onFlush',
			'postUpdate',
		];
	}

	public function postPersist(LifecycleEventArgs $args): void
	{
		$entity = $args->getObject();
		if ($entity instanceof OrderItem && $entity->getProductId()) {
			self::$update[] = $entity->getProductId();
		}
	}

	public function onFlush(OnFlushEventArgs $args): void
	{
		$uow = $this->em->getUnitOfWork();

		foreach ($uow->getScheduledEntityUpdates() as $object) {
			if ($object instanceof Product) {
				$changeSet = $uow->getEntityChangeSet($object);
				unset($changeSet['modified'], $changeSet['productTexts']);
				$changeSetCount = count($changeSet);
				if ($changeSetCount === 0) {
					continue;
				}
				self::$update[] = $object->getId();
			} else if ($object instanceof ProductTexts) {
				$changeSet      = $uow->getEntityChangeSet($object);
				$changeSetCount = count($changeSet);
				if ($changeSetCount === 0) {
					continue;
				}

				if ($object->getProduct() instanceof Product) {
					self::$update[] = $object->getProduct()->getId();
				}
			}
		}

		foreach ($uow->getScheduledEntityDeletions() as $object) {
			if ($object instanceof Product === false) {
				continue;
			}

			foreach ($this->clients->getClients() as $client) {
				try {
					if ($this->eshopProducts->isProductAllowed((int) $object->getId(), $client)) {
						$this->productsApi->deleteProduct($client, (string) $object->getId());
					}
				} catch (Exception) {
				}
			}
		}
	}

	public function postUpdate(LifecycleEventArgs $event): void
	{
		try {
			$this->checkProductQuantity($event->getObject());
		} catch (Exception) {
		}
	}

	protected function checkProductQuantity(object $entity): void
	{
		if ($entity instanceof Product === false) {
			return;
		}

		// Pokud je ulozen productForm, tak ignorujeme aktualizaci
		if (isset($_REQUEST['_do']) && $_REQUEST['_do'] === 'productForm-form-submit') {
			return;
		}

		self::$update[] = $entity->getId();
	}
}
