<?php declare(strict_types = 1);

namespace MultihubDropShip\Model\Sync;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Arrays;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Invoices;
use MultihubDropShip\Model\Api\OrderApi;
use MultihubDropShip\Model\Clients;
use MultihubDropShip\Model\Dao\Client;
use MultihubDropShip\Model\Entities\DropShipOrder;
use MultihubDropShip\Model\MultihubDropShipLogger;
use MultihubDropShip\Model\Orders;

class SyncOrder
{
	public function __construct(
		protected Clients                $clients,
		protected EntityManagerDecorator $em,
		protected OrderApi               $orderApi,
		protected Orders                 $orders,
		protected Invoices               $invoices,
	)
	{
	}

	public function downloadNew(Client $client): void
	{
		foreach ($this->orderApi->getNewOrders($client) as $row) {
			$uploadInvoices = [];
			$dropShipOrder  = $this->orders->addOrder($client, $row);

			if ($dropShipOrder) {
				$setAccepted = $this->orderApi->setAccepted($client, $dropShipOrder->dropShipId);

				if ($setAccepted) {
					$dropShipOrder->lastStatus = DropShipOrder::StatusAccepted;
					$this->em->persist($dropShipOrder);
					$this->em->flush();

					if (
						EshopOrdersConfig::load('invoice.enable')
						&& $dropShipOrder->order->isPaid
						&& Arrays::contains(EshopOrdersConfig::loadArray('invoice.orderStatusSendTrigger') ?: [], 'isPaid')
					) {
						$uploadInvoices[] = $dropShipOrder;
					}
				} else {
					MultihubDropShipLogger::log('Set accepted error ' . $dropShipOrder->dropShipId);
					LogNotifier::toDevelopers('Set accepted error ' . $dropShipOrder->dropShipId, 'multihub dropship');
				}
			}

			sleep(1);
			$this->em->clear();
			foreach ($uploadInvoices as $dropShipOrder) {
				$this->invoices->getOneByOrder($dropShipOrder->order, true);
			}
		}
	}

	public function downloadCancelled(Client $client): void
	{
		foreach ($this->orderApi->getCancelled($client) as $row) {
			$dropShipOrder = $this->orders->getOrderByDropShipId($row['id']);

			if ($dropShipOrder && $dropShipOrder->lastStatus === DropShipOrder::StatusAccepted) {
				$this->orders->setCancelled($dropShipOrder);
			}
		}
	}

	public function getDetail(Client $client, string $dropShipId): ?array
	{
		return $this->orderApi->getDetail($client, $dropShipId);
	}
}
