<?php declare(strict_types = 1);

namespace MyGls\Model;

use Gls\Model\Exception;
use GuzzleHttp\Client;
use Nette\Utils\Json;
use Tracy\Debugger;

class Api
{
	public function printLabels($username, $password, $url, $orderId, $parcels): ?array
	{
		$method  = 'PrintLabels';
		$request = $this->getRequestString($username, $password, $parcels, false, $method);

		$response = $this->getResponse($url, $method, $request, false);
		if ($response == true) {
			$oResponse = $response;
			$response  = Json::decode($response);

			if (count($response->PrintLabelsErrorList) === 0 && count($response->Labels) > 0) {
				$file = MYGLS_DIR . '/labels/' . $orderId . '.pdf';
				file_put_contents($file, implode('', array_map('chr', $response->Labels)));
				$response->Labels = null;

				$result = [];
				foreach ($response->PrintLabelsInfoList as $row) {
					$result[] = $row->ParcelNumber;
				}

				return $result;
			} else {
				Debugger::log($oResponse, 'myGlsPrintLabels');
				Debugger::log($parcels, 'myGlsPrintLabels');

				foreach ($response->PrintLabelsErrorList as $row) {
					throw new \Exception($row->ErrorDescription);
				}
			}
		}

		return null;
	}

	/*
	* Preparing label(s) by the service.
	*/
	public function prepareLabels($username, $password, $url, $parcels): array
	{
		$method  = 'PrepareLabels';
		$request = $this->getRequestString($username, $password, $parcels, false, $method);

		$response = $this->getResponse($url, $method, $request, false);

		$result = [];
		if ($response) {
			$response = Json::decode($response);
			if (count($response->PrepareLabelsError) === 0 && count($response->ParcelInfoList) > 0) {
				foreach ($response->ParcelInfoList as $row) {
					$result[(int) $row->ClientReference] = $row->ParcelId;
				}
			}
		}

		return $result;
	}

	/*
	* Get parcel(s) information by date ranges.
	*/
	public function getParcelList($username, $password, $url, $method, $isXmlFormat)
	{
		//Test request:
		if ($isXmlFormat == false) {
			$pickupDateFrom = "\/Date(" . (strtotime("2020-04-16 23:59:59") * 1000) . ")\/";
			$pickupDateTo   = "\/Date(" . (strtotime("2020-04-16 23:59:59") * 1000) . ")\/";
			$request        = '{"Username":"' . $username . '","Password":' . $password . ',"PickupDateFrom":"' . $pickupDateFrom . '","PickupDateTo":"' . $pickupDateTo . '","PrintDateFrom":null,"PrintDateTo":null}';
		} else {
			$request = '<GetParcelListRequest
					xmlns="http://schemas.datacontract.org/2004/07/GLS.MyGLS.ServiceData.APIDTOs.ParcelOperations"
					xmlns:i="http://www.w3.org/2001/XMLSchema-instance">
						<ClientNumberList
						xmlns="http://schemas.datacontract.org/2004/07/GLS.MyGLS.ServiceData.APIDTOs.Common"
						xmlns:a="http://schemas.microsoft.com/2003/10/Serialization/Arrays"/>
						<Password
							xmlns="http://schemas.datacontract.org/2004/07/GLS.MyGLS.ServiceData.APIDTOs.Common">' . $password . '</Password>
						<Username
							xmlns="http://schemas.datacontract.org/2004/07/GLS.MyGLS.ServiceData.APIDTOs.Common">' . $username . '</Username>
						<PickupDateFrom>2020-04-16T00:00:00</PickupDateFrom>
						<PickupDateTo>2020-04-16T00:00:00</PickupDateTo>
						<PrintDateFrom i:nil="true"/>
						<PrintDateTo i:nil="true"/>
					</GetParcelListRequest>';
		}

		$response = $this->getResponse($url, $method, $request, $isXmlFormat);

		if ($isXmlFormat == false) {
			var_dump(count(json_decode((string) $response)->GetParcelListErrors));
			var_dump(count(json_decode((string) $response)->PrintDataInfoList));
		} else if ($response != "") {
			die("GetParcelList: OK");
			//die($response);
			/*
			//https://www.php.net/manual/en/public function.xml-parse-into-struct.php
			$p = xml_parser_create();
			xml_parse_into_struct($p, $response, $vals, $index);
			xml_parser_free($p);

			foreach ($vals as $val)
			{
				var_dump($val);
			}
			*/
		}
	}

	/*
	* Get parcel statuses.
	*/
	public function getParcelStatuses($username, $password, $url, $method, $isXmlFormat)
	{
		//Test request:
		$parcelNumber = 0;
		if ($isXmlFormat == false) {
			$request = '{"Username":"' . $username . '","Password":' . $password . ',"ParcelNumber":' . $parcelNumber . ',"ReturnPOD":true,"LanguageIsoCode":"HU"}';
		} else {
			$request = '<GetParcelStatusesRequest
					xmlns="http://schemas.datacontract.org/2004/07/GLS.MyGLS.ServiceData.APIDTOs.ParcelOperations"
					xmlns:i="http://www.w3.org/2001/XMLSchema-instance">
						<ClientNumberList
						xmlns="http://schemas.datacontract.org/2004/07/GLS.MyGLS.ServiceData.APIDTOs.Common"
						xmlns:a="http://schemas.microsoft.com/2003/10/Serialization/Arrays"/>
						<Password
							xmlns="http://schemas.datacontract.org/2004/07/GLS.MyGLS.ServiceData.APIDTOs.Common">' . $password . '</Password>
						<Username
							xmlns="http://schemas.datacontract.org/2004/07/GLS.MyGLS.ServiceData.APIDTOs.Common">' . $username . '</Username>
						<LanguageIsoCode>HU</LanguageIsoCode>
						<ParcelNumber>' . $parcelNumber . '</ParcelNumber>
						<ReturnPOD>true</ReturnPOD>
					</GetParcelStatusesRequest>';
		}

		$response = $this->getResponse($url, $method, $request, $isXmlFormat);

		if ($isXmlFormat == false) {
			if ($response == true) {
				var_dump(count(json_decode((string) $response)->GetParcelStatusErrors));
				if (count(json_decode((string) $response)->GetParcelStatusErrors) === 0 && count(json_decode((string) $response)->POD) > 0) {
					//POD saving:
					$pdf = implode('', array_map('chr', json_decode((string) $response)->POD));

					file_put_contents('php_rest_client_' . $method . '.pdf', $pdf);
				}
			}
		} else if ($response != "" && !str_contains((string) $response, "ErrorCode")) {
			$podNodeName = "<POD>";
			$pod         = strpos((string) $response, $podNodeName);
			if ($pod != false) {
				$firstPosition = ($pod + strlen($podNodeName));
				$lastPosition  = strpos((string) $response, "</POD>");

				//POD saving:
				file_put_contents('php_rest_client_' . $method . '.pdf', base64_decode(substr((string) $response, $firstPosition, ($lastPosition - $firstPosition))));
			}
		}
	}

	//Utility public functions:

	public function getRequestString($username, $password, $dataList, $isXmlFormat, $requestObjectName)
	{
		$result = "";

		if ($isXmlFormat == false) {
			switch ($requestObjectName) {
				case "PrintLabels":
					$result = '{"Username":"' . $username . '","Password":' . $password . ',"ParcelList":' . $dataList . ',"ShowPrintDialog":0}';
					break;
				case "PrepareLabels":
					$result = '{"ClientNumberList": null, "Username":"' . $username . '","Password":' . $password . ',"ParcelList":' . $dataList . '}';
					break;
				case "GetPrintedLabels":
					$result = '{"Username":"' . $username . '","Password":' . $password . ',"ParcelIdList":' . $dataList . ',"ShowPrintDialog":0}';;
					break;
			}
		} else {
			$additionalParameters = "";
			if (in_array($requestObjectName, ["PrintLabels", "GetPrintedLabels"])) {
				$additionalParameters = "<PrintPosition>1</PrintPosition><ShowPrintDialog>false</ShowPrintDialog>";
			}

			$result = '<' . $requestObjectName . 'Request
		xmlns="http://schemas.datacontract.org/2004/07/GLS.MyGLS.ServiceData.APIDTOs.LabelOperations"
		xmlns:i="http://www.w3.org/2001/XMLSchema-instance">
			<ClientNumberList
			xmlns="http://schemas.datacontract.org/2004/07/GLS.MyGLS.ServiceData.APIDTOs.Common"
			xmlns:a="http://schemas.microsoft.com/2003/10/Serialization/Arrays"/>
			<Password
				xmlns="http://schemas.datacontract.org/2004/07/GLS.MyGLS.ServiceData.APIDTOs.Common">' . $password . '</Password>
			<Username
				xmlns="http://schemas.datacontract.org/2004/07/GLS.MyGLS.ServiceData.APIDTOs.Common">' . $username . '</Username>
			' . $dataList . $additionalParameters . '
		</' . $requestObjectName . 'Request>';

			$toBeReplaced = ["\r", "\n"];
			$result       = str_replace($toBeReplaced, "", $result);
		}

		return $result;
	}

	public function getResponse($url, $method, $request, $isXmlFormat)
	{
		//Service calling:
		$curl = curl_init();
		curl_setopt($curl, CURLOPT_POST, 1);
		curl_setopt($curl, CURLOPT_URL, $url . $method);
		curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
		curl_setopt($curl, CURLOPT_TIMEOUT, 600);
		curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, 0);
		curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, 0);
		curl_setopt($curl, CURLOPT_POSTFIELDS, $request);
		if ($isXmlFormat == false) {
			curl_setopt($curl, CURLOPT_HTTPHEADER, [
					'Content-Type: application/json',
					'Content-Length: ' . strlen((string) $request)]
			);
		} else {
			curl_setopt($curl, CURLOPT_HTTPHEADER, [
					'Content-Type: text/xml',
					'Content-Length: ' . strlen((string) $request)]
			);
		}

		$response = curl_exec($curl);

		if (curl_getinfo($curl)["http_code"] == "401") {
			die("Unauthorized");
		}

		if ($response === false) {
			die('curl_error:"' . curl_error($curl) . '";curl_errno:' . curl_errno($curl));
		}
		curl_close($curl);

		return $response;
	}

	public function getParcelStatusList($parcelNumber)
	{
		$config_array = [
			'verify' => false,
			'debug'  => false,
		];
		$client       = new Client($config_array);
		$response     = $client->request("GET", $this->getTrackingUrlXml($parcelNumber));
		$xml          = simplexml_load_string((string) $response->getBody());

		try {
			if ($xml !== false) {
				$arr = [];
				foreach ($xml->Parcel->Statuses->Status as $status)
					$arr[] = ((array) $status)['@attributes'];

				return $arr;
			} else {
				throw new Exception('Tracking code wasn`t registered or error occured!');
			}
		} catch (\Exception) {
		}

		return false;
	}

	public function getTrackingUrlXml($parcelNumber): string
	{
		return "http://online.gls-czech.com/tt_page_xml.php?pclid=$parcelNumber";
	}
}
