<?php declare(strict_types = 1);

namespace MyGls\AdminModule\Model\Subscribers;

use Contributte\Translation\Translator;
use EshopOrders\Model\EshopOrdersConfig;
use MyGls\Model\MyGlsConfig;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Core\Model\Event\ControlEvent;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Model\Event\OrderNavigationCallbackEvent;
use EshopOrders\AdminModule\Model\Event\OrdersNavigationEvent;
use EshopOrders\AdminModule\Model\Speditions;
use MyGls\Model\OrderApiService;
use MyGls\Model\OrdersExported;

class GlsOrdersSubscriber implements EventSubscriberInterface
{
	protected OrdersExported  $ordersExported;
	protected OrderApiService $apiService;
	protected Translator      $translator;
	protected Speditions      $speditionsService;

	public function __construct(
		OrdersExported  $ordersExported,
		OrderApiService $apiService,
		Translator      $translator,
		Speditions      $speditionsService
	)
	{
		$this->ordersExported    = $ordersExported;
		$this->apiService        = $apiService;
		$this->translator        = $translator;
		$this->speditionsService = $speditionsService;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.admin.ordersDefaultNavigation'         => 'ordersDefaultNavigation',
			'eshopOrders.admin.ordersDefaultNavigationCallback' => 'ordersDefaultNavigationCallback',
			OrdersGrid::class . '::onAttach'                    => ['ordersGridAttached', 255],
		];
	}

	public function ordersGridAttached(ControlEvent $event): void
	{
		/** @var OrdersGrid $control */
		$control = $event->control;

		$control->addExportCallback('myGls', 'myGls.ordersGrid.export', function(array $ids, $quantity) use ($control) {
			$orders = $this->ordersExported->getOrdersNotExported($ids);
			if (!$orders) {
				$control->presenter->flashMessageSuccess($this->translator->translate('myGls.export.noOrders'));
				$control->presenter->redrawControl('flashes');

				return;
			}

			set_time_limit(120);
			$result = $this->apiService->sendOrders($orders, max(1, (int) $quantity));

			if ($result['error'] === 0) {
				$control->presenter->flashMessageSuccess($this->translator->translate('myGls.export.success'));
			} else if ($result['error'] > 0 && $result['ok'] > 0) {
				$control->presenter
					->flashMessageWarning($this->translator->translate('myGls.export.partial', null, $result));
			} else {
				$control->presenter->flashMessageDanger($this->translator->translate('myGls.export.error'));
			}

			$control->presenter->redrawControl('flashes');
		});

		if (MyGlsConfig::load('allowGenerateLabel', false)) {
			$control->addExportCallback('myGlsLabels', 'myGls.ordersGrid.labels', function(array $ids) use ($control) {
				$orders = $this->ordersExported->getOrdersExported($ids);
				if (!$orders) {
					$control->presenter->flashMessageSuccess($this->translator->translate('myGls.labels.noOrders'));
					$control->presenter->redrawControl('flashes');

					return;
				}

				set_time_limit(120);
				$result = $this->apiService->generateLabels($orders);

				if ($result['error'] === 0) {
					foreach ($result['files'] as $file) {
						$arr                                         = [
							'name' => 'MY GLS labels',
							'url'  => $control->presenter->link(':Core:Admin:FileResponse:', base64_encode(serialize([
								'file'     => $file,
								'filename' => basename($file),
							]))),
						];
						$control->presenter->payload->fileRequests[] = $arr;
					}

					$control->presenter->flashMessageSuccess($this->translator->translate('myGls.labels.success'));
				} else {
					$control->presenter->flashMessageDanger($this->translator->translate('myGls.labels.error'));
				}

				$control->presenter->redrawControl('flashes');
			});
		}
	}

	public function ordersDefaultNavigation(OrdersNavigationEvent $event): void
	{
		if (EshopOrdersConfig::load('hideOrderGridNavExport')) {
			return;
		}

		$event->addNavigation(
			'myGls.ordersNavigation.exportAll',
			['navigationCallback!', ['key' => 'myGlsExport']],
			'fas fa-truck-loading',
			'ajax'
		);
	}

	public function ordersDefaultNavigationCallback(OrderNavigationCallbackEvent $event): void
	{
		if ($event->key !== 'myGlsExport') {
			return;
		}

		$orders = $this->ordersExported->getOrdersNotExported();

		$ids = [];
		foreach ($orders as $k => $order) {
			$ids[] = $order->getOrder()->getId();
		}

		if ($ids) {
			$allowed = $this->speditionsService->checkOrdersGridMenuExportOnlyInStatus($ids);

			foreach ($orders as $k => $order) {
				if (!in_array($order->getOrder()->getId(), $allowed)) {
					unset($orders[$k]);
				}
			}
		}

		if (!$orders) {
			$event->presenter->flashMessageSuccess($this->translator->translate('myGls.export.noOrders'));
			$event->presenter->redrawControl('flashes');

			return;
		}

		set_time_limit(120);
		$result = $this->apiService->sendOrders($orders);

		if ($result['error'] === 0) {
			$event->presenter->flashMessageSuccess($this->translator->translate('myGls.export.success'));
		} else if ($result['error'] > 0 && $result['ok'] > 0) {
			$event->presenter->flashMessageWarning($this->translator->translate('myGls.export.partial', null, $result));
		} else {
			$event->presenter->flashMessageDanger($this->translator->translate('myGls.export.error'));
		}

		$event->presenter->redrawControl('flashes');
	}
}
