<?php declare(strict_types = 1);

namespace MyGls\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\GridFilterEvent;
use Core\Model\Event\SetFormDataEvent;
use Doctrine\Common\Collections\Criteria;
use EshopOrders\AdminModule\Components\Order\OrderForm;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Components\Order\OrderSpeditionForm;
use EshopOrders\AdminModule\Components\Order\OrderStatusesGrid;
use EshopOrders\AdminModule\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use Exception;
use MyGls\Model\Entities\GlsOrder;
use MyGls\Model\Entities\GlsParcelNumber;
use MyGls\Model\OrdersExported;
use Nette\Utils\Html;
use Nette\Utils\Strings;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class OrderSubscriber implements EventSubscriberInterface
{
	/** @var GlsOrder[]|null */
	protected static ?array          $allOrders = null;
	protected OrdersExported         $ordersExported;
	protected EntityManagerDecorator $em;
	protected EventDispatcher        $eventDispatcher;

	public function __construct(
		OrdersExported         $ordersExported,
		EntityManagerDecorator $em,
		EventDispatcher        $eventDispatcher
	)
	{
		$this->ordersExported  = $ordersExported;
		$this->em              = $em;
		$this->eventDispatcher = $eventDispatcher;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			OrderForm::class . '::onAttach'               => 'orderFormAttached',
			OrdersGrid::class . '::columnSpeditionRender' => 'columnSpeditionRender',
			OrdersGrid::class . '::filterPackageNumber'   => 'gridFilterPackageNumber',
			OrderSpeditionForm::class . '::createForm'    => 'orderSpeditionCreateForm',
			OrderSpeditionForm::class . '::formSuccess'   => 'orderSpeditionFormSuccess',
			OrderSpeditionForm::class . '::setOrder'      => 'orderSpeditionSetOrder',
			OrderStatusesGrid::class . '::beforeDelete'   => 'orderStatusDelete',
		];
	}

	public function orderFormAttached(ControlEvent $event): void
	{
		/** @var OrderForm $control */
		$control   = $event->control;
		$order     = $control->order;
		$spedition = $order->getSpedition() ? $order->getSpedition()->getSpedition() : null;

		if ($spedition && $spedition->getIdent() === 'myGls') {
			$control->template->glsExport                   = $this->ordersExported->getOrdersExported([$order->getId()])[$order->getId()] ?? null;
			$control->template->speditionExportInfoTemplate = __DIR__ . '/exportInfoTemplate.latte';
		}
	}

	public function columnSpeditionRender(Event $event): void
	{
		$data           = $event->data;
		$speditionIdent = 'myGls';
		$export         = null;

		/** @var Order $order */
		$order = $data['order'];

		/** @var Html $html */
		$html = $data['html'];

		if ($data['ordersBySpedition'][$speditionIdent] ?? null) {
			if (self::$allOrders === null) {
				self::$allOrders = $this->ordersExported->getOrdersExported(array_keys($data['ordersBySpedition'][$speditionIdent] ?? []), false);
			}

			$export = self::$allOrders[$order->getId()] ?? null;
		} else if ($order->getSpeditionIdent() === $speditionIdent) {
			$export = $this->ordersExported->getOrdersExported([$order->getId()], false)[$order->getId()] ?? null;
		}

		if ($export) {
			$h     = $html->addHtml(Html::el('div'));
			$urls  = $export->getTrackingUrls();
			$count = count($urls);
			$i     = 1;
			foreach ($urls as $number => $url) {
				$h->addHtml(Html::el('a', ['target' => '_blank'])
					->setAttribute('href', $url)
					->setText($number)
				);
				if ($i !== $count) {
					$h->addText(', ');
				}
				$i++;
			}
		}
	}

	public function gridFilterPackageNumber(GridFilterEvent $event): void
	{
		$ids = $this->ordersExported->findIdByPackageNumber($event->value);

		if ($ids) {
			$event->criteria->orWhere(Criteria::expr()->in('o.id', $ids));
		}
	}

	public function orderSpeditionCreateForm(CreateFormEvent $event): void
	{
		/** @var OrderSpeditionForm $control */
		$control = $event->control;
		$form    = $event->form;

		$glsIds = [];
		foreach ($control->getSpeditions() as $sped) {
			if ($sped->getIdent() === 'myGls') {
				$glsIds[] = $sped->getId();
			}
		}

		if (empty($glsIds)) {
			return;
		}

		$container = $form->addContainer('myGls', '');
		$container->addText('numberPackage', 'myGls.entity.numberPackage')
			->setDescription('myGls.entity.numberPackageDesc');
		$container->addText('parcelNumber', 'myGls.entity.parcelNumber');
		$container->addBool('removeExport', 'myGls.entity.removeExportStatus')
			->setDefaultValue(0);
		$container->addBool('updateDeliveryAddress', 'myGls.entity.updateDeliveryAddress')
			->setDefaultValue(0);

		$form->getComponent('spedition')->addCondition($form::IS_IN, $glsIds)
			->toggle($form->getComponent('myGls')['numberPackage']->getHtmlId())
			->toggle($form->getComponent('myGls')['removeExport']->getHtmlId())
			->toggle($form->getComponent('myGls')['updateDeliveryAddress']->getHtmlId())
			->toggle($form->getComponent('myGls')['parcelNumber']->getHtmlId());
	}

	public function orderSpeditionFormSuccess(FormSuccessEvent $event): void
	{
		/** @var OrderSpedition $orderSpedition */
		$orderSpedition = $event->custom['entity'];
		/** @var OrderSpeditionForm $control */
		$control  = $event->control;
		$orderId  = $orderSpedition->getOrder()->getId();
		$values   = $event->values;
		$glsOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;

		$isGls = false;
		foreach ($control->getSpeditions() as $sped) {
			if ($sped->getIdent() === 'myGls' && $sped->getId() == $values->spedition) {
				$isGls = true;
				break;
			}
		}

		try {
			if ($isGls) {
				if (!$glsOrder) {
					$glsOrder = new GlsOrder($orderSpedition->getOrder());
				}

				$numberPackages      = trim(str_replace(' ', '', (string) $values->myGls->numberPackage)) ?: null;
				$numberPackages      = $numberPackages ? trim($numberPackages, ',') : null;
				$numberPackages      = $numberPackages && Strings::contains($numberPackages, ',') ? array_map('trim', explode(',', $numberPackages)) : [];
				$numberPackagesCount = count($numberPackages);

				$glsOrder->parcelNumber = $values->myGls->parcelNumber ?: null;

				if ($numberPackagesCount <= 1) {
					$glsOrder->numberPackage = $numberPackages[0] ?? null;
					$glsOrder->associatedNumberPackages->clear();
				} else {
					$existsPackages            = $glsOrder->associatedNumberPackages->getKeys();
					$existsPackagesWithoutMain = array_slice($numberPackages, 1, count($numberPackages) - 1);
					$diffToAdd                 = array_diff($existsPackagesWithoutMain, $existsPackages);
					$diffToRemove              = array_diff($existsPackages, $existsPackagesWithoutMain);

					for ($i = 0; $i < $numberPackagesCount; $i++) {
						$numberPackage = $numberPackages[$i];
						if ($i === 0) {
							$glsOrder->numberPackage = $numberPackage;
						} else {
							if (in_array($numberPackage, $diffToAdd)) {
								$glsOrder->associatedNumberPackages->add(new GlsParcelNumber($numberPackage, $glsOrder));
							}
						}
					}

					foreach ($diffToRemove as $p) {
						$glsOrder->associatedNumberPackages->remove($p);
					}
				}

				if ($glsOrder->numberPackage && $glsOrder->getExported() === null) {
					$glsOrder->export();
				}

				if ($values->myGls->removeExport === 1) {
					$glsOrder->resetExport();
				}

				$this->em->persist($glsOrder);

				if ($event->presenter && $values['myGls']->updateDeliveryAddress) {
					$event->presenter->flashMessageWarning('myGls.orderForm.updateDeliveryAddress');
					$event->presenter->redrawControl('flashes');
				}

				$this->eventDispatcher->dispatch(new Event(['order'          => $glsOrder->getOrder(),
				                                            'numberPackages' => $numberPackages]), 'orderSubscriber.orderSpeditionFormSuccess');
			} else if ($glsOrder) {
				$this->em->remove($glsOrder);
			}
		} catch (Exception $e) {
			Debugger::log($e, 'orderSpeditionForm');
			$event->presenter->flashMessageDanger('myGls.orderForm.speditionSaveError');
		}
	}

	public function orderSpeditionSetOrder(SetFormDataEvent $event): void
	{
		/** @var OrderSpedition $orderSpedition */
		$orderSpedition = $event->entity;
		$orderId        = $orderSpedition->getOrder()->getId();

		$glsOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;
		if (!$glsOrder) {
			return;
		}

		$event->form->getComponent('myGls')->setDefaults([
			'parcelNumber'  => $glsOrder->parcelNumber,
			'numberPackage' => implode(', ', $glsOrder->getAllNumberPackages()),
		]);
	}

	public function orderStatusDelete(OrderStatusEvent $event): void
	{
		$orderStatus = $event->orderStatus;
		$order       = $orderStatus->getOrder();

		if ($orderStatus->getStatus()->getId() !== OrderStatus::STATUS_SPEDITION || ($order->getSpeditionIdent() && $order->getSpeditionIdent() !== 'myGls')) {
			return;
		}

		$orderId  = $order->getId();
		$glsOrder = $this->ordersExported->getOrders(null, [$orderId])[$orderId] ?? null;

		if (!$glsOrder) {
			return;
		}

		$glsOrder->resetExport();
		$this->em->persist($glsOrder);
		$this->em->flush($glsOrder);
	}
}
