<?php declare(strict_types = 1);

namespace MyGls\Model\Entities;

use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use MyGls\Model\MyGlsConfig;
use Nette\Utils\DateTime;
use EshopOrders\Model\Entities\Order;

/**
 * @ORM\Table(name="my_gls__order")
 * @ORM\Entity
 * @ORM\EntityListeners({"EshopOrders\Model\Listeners\CarrierListener"})
 */
class GlsOrder
{
	public const STATUS_COMPLETED = 'completed';
	public const STATUS_NOT_FOUND = 'notFound';
	public const STATUS_RETURNED  = 'returned';

	/**
	 * @var Order
	 * @ORM\Id
	 * @ORM\OneToOne(targetEntity="\EshopOrders\Model\Entities\Order")
	 * @ORM\JoinColumn(name="order_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected Order $order;

	/**
	 * @ORM\Column(type="string", length=30, nullable=true)
	 */
	public ?string $parcelNumber = null;

	/**
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	private ?DateTimeInterface $exported = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $numberPackage = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $lastStatus;

	/**
	 * @var Collection<string, GlsParcelNumber>
	 * @ORM\OneToMany(targetEntity="GlsParcelNumber", mappedBy="parcelOrder", cascade={"all"}, orphanRemoval=true, indexBy="numberPackage")
	 */
	public Collection $associatedNumberPackages;

	public function __construct(Order $order)
	{
		$this->order                    = $order;
		$this->associatedNumberPackages = new ArrayCollection;
	}

	public function getOrder(): Order { return $this->order; }

	public function export(?DateTime $datetime = null): self
	{
		$this->exported = $datetime ?? new DateTime;

		return $this;
	}

	public function getExported(): ?DateTimeInterface { return $this->exported; }

	public function getTrackingUrl(): string
	{
		return sprintf(MyGlsConfig::load('trackingUrl', ''), $this->numberPackage);
	}

	public function getTrackingUrls(): array
	{
		$result = [];
		if (empty(!$this->getTrackingUrl())) {
			$result[$this->numberPackage] = $this->getTrackingUrl();
		}
		foreach ($this->getAssociatedNumberPackages() as $number) {
			$result[$number->numberPackage] = sprintf(MyGlsConfig::load('trackingUrl', ''), $number->numberPackage);
		}

		return $result;
	}

	public function resetExport(): void
	{
		$this->numberPackage = null;
		$this->lastStatus    = null;
		$this->exported      = null;

		$this->associatedNumberPackages->clear();
	}

	public function getAssociatedNumberPackages(): array
	{
		return $this->associatedNumberPackages->toArray();
	}

	public function getAllNumberPackages(): array
	{
		if (!$this->numberPackage) {
			return [];
		}

		return array_merge([$this->numberPackage], array_map(static fn(GlsParcelNumber $parcel) => $parcel->getNumberPackage(), $this->getAssociatedNumberPackages()));
	}

}
