<?php declare(strict_types = 1);

namespace MyGls\Model;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\Model\ExpeditionLogger;
use EshopOrders\Model\Helpers\OrderHelper;
use Exception;
use Gls\Model\Exception\ParcelLabels;
use MyGls\Model\Entities\GlsOrder;
use MyGls\Model\Entities\GlsParcelNumber;
use Nette\Utils\DateTime;
use Nette\Utils\Json;
use stdClass;
use Tracy\Debugger;

class OrderApiService
{
	public const LABELS_DIR = TMP_DIR . '/eshopOrders/myGlsLabels/';
	protected EntityManagerDecorator $em;
	protected Sellers                $sellers;
	protected Translator             $translator;
	protected ExpeditionLogger       $expeditionLogger;

	public function __construct(
		EntityManagerDecorator $em,
		Sellers                $sellers,
		Translator             $translator,
		ExpeditionLogger       $expeditionLogger
	)
	{
		$this->em               = $em;
		$this->sellers          = $sellers;
		$this->translator       = $translator;
		$this->expeditionLogger = $expeditionLogger;
	}

	/**
	 * TODO logovani do db
	 *
	 * @param GlsOrder[] $orders
	 *
	 * @return int[]
	 * @throws Exception
	 */
	public function sendOrders(array $orders, int $quantity = 1): array
	{
		$api              = new Api;
		$clientNumber     = (string) MyGlsConfig::load('clientNumber');
		$username         = (string) MyGlsConfig::load('username');
		$password         = "[" . implode(
				',',
				(array) unpack('C*', hash('sha512', (string) MyGlsConfig::load('password'), true))
			) . "]";
		$urlParcelService = (string) MyGlsConfig::load('urlParcelService');
		$orderIds         = [];
		$result           = [
			'ok'    => 0,
			'error' => 0,
		];
		$parcelNumbers    = [];

		foreach ($orders as $glsOrder) {
			$parcels = [];
			$order   = $glsOrder->getOrder();
			$address = $order->getAddressDelivery();
			$addrInv = $order->getAddressInvoice();
			$seller  = $this->sellers->getSellerForSite($order->site->getIdent());
			$cod     = 0;

			if ($order->getPaymentIdent() === 'cod') {
				$curr = $order->getCurrencyCode();
				$cod  = round($order->getPrice(true), $curr === 'CZK' ? 0 : $order->currency->decimals);

				if (
					(!$address && $curr === 'EUR')
					|| ($address && Strings::lower($address->getCountry()->getId()) === 'sk')
				) {
					$cod = OrderHelper::roundSkCod($cod);
				}
			}

			$formattedPhone    = str_replace(' ', '', Strings::phoneFormat($address->getPhone(), strtoupper($address->getCountry()->getId())));
			$formattedInvPhone = str_replace(' ', '', Strings::phoneFormat($addrInv->getPhone(), strtoupper($addrInv->getCountry()->getId())));

			$parcel                  = new stdClass;
			$parcel->ClientNumber    = (int) $clientNumber;
			$parcel->ClientReference = (string) $order->getId();
			$parcel->CODAmount       = (float) $cod;
			$parcel->CODCurrency     = strtoupper($order->getCurrencyCode());
			$parcel->CODReference    = $order->getId();
			$parcel->Content         = $this->translator->translate('myGls.labels.apiContent') . $order->getId();
			$parcel->Count           = $quantity;
			$parcel->PickupDate      = "/Date(" . (new DateTime)->setTime(0, 0, 0)->getTimestamp() . ")/";

			$sender                  = new stdClass;
			$sender->City            = $seller->city2;
			$sender->ContactEmail    = $seller->email;
			$sender->ContactName     = $seller->name;
			$sender->ContactPhone    = str_replace(' ', '', $seller->phone);
			$sender->CountryIsoCode  = strtoupper($seller->country2->getId());
			$sender->HouseNumber     = '';
			$sender->Name            = $seller->name;
			$sender->Street          = $seller->street2;
			$sender->ZipCode         = $seller->postal2;
			$sender->HouseNumberInfo = '';
			$parcel->PickupAddress   = $sender;

			$deliveryAddress                  = new stdClass;
			$deliveryAddress->City            = $address->getCity();
			$deliveryAddress->ContactEmail    = $address->getEmail() ?: $addrInv->getEmail();
			$deliveryAddress->ContactName     = $address->getName() ?: $addrInv->getName();
			$deliveryAddress->ContactPhone    = $formattedPhone ?: $formattedInvPhone;
			$deliveryAddress->CountryIsoCode  = strtoupper($address->getCountry()->getId());
			$deliveryAddress->HouseName       = '';
			$deliveryAddress->Name            = $address->getName();
			$deliveryAddress->Street          = $address->getStreet();
			$deliveryAddress->ZipCode         = $address->getPostal();
			$deliveryAddress->HouseNumberInfo = '';
			$parcel->DeliveryAddress          = $deliveryAddress;

			$servicesList = [];

			if ($glsOrder->parcelNumber) {
				$service               = new stdClass;
				$service->Code         = 'PSD';
				$param                 = new stdClass;
				$param->StringValue    = $glsOrder->parcelNumber;
				$service->PSDParameter = $param;
				$servicesList[]        = $service;
			} else {
				$service               = new stdClass;
				$service->Code         = 'FDS';
				$param                 = new stdClass;
				$param->StringValue    = $address->getEmail();
				$service->FDSParameter = $param;
				$servicesList[]        = $service;

				if (Arrays::contains(['cs', 'sk'], Strings::lower($address->getCountry()->getId()))) {
					$service               = new stdClass;
					$service->Code         = 'FSS';
					$param                 = new stdClass;
					$param->StringValue    = $formattedPhone;
					$service->FSSParameter = $param;
					$servicesList[]        = $service;
				}
			}

			$parcel->ServiceList = $servicesList;

			$parcels[] = $parcel;

			$orderIds[$order->getId()] = $order->getId();

			try {
				$parcelNumbers[$order->getId()] = $api->printLabels($username, $password, $urlParcelService, $order->getId(), Json::encode($parcels));
			} catch (Exception $e) {
				$this->expeditionLogger->logError('myGls', $e->getMessage(), $order->getId());
			}
		}

		foreach ($parcelNumbers as $orderId => $numbers) {
			if (!$numbers) {
				continue;
			}

			$firstNumber = array_shift($numbers);
			/** @var GlsOrder $entity */
			$entity                = $this->em->getReference(GlsOrder::class, $orderId);
			$entity->numberPackage = (string) $firstNumber;
			$entity->export();
			$this->em->persist($entity);

			foreach ($numbers as $number) {
				$pn = new GlsParcelNumber((string) $number, $entity);
				$this->em->persist($pn);
			}

			unset($orderIds[$orderId]);
			$result['ok']++;
		}
		$this->em->flush();

		$result['error'] = count($orderIds);

		return $result;
	}

	/**
	 * @param GlsOrder[] $orders
	 *
	 * @throws ParcelLabels
	 */
	public function generateLabels(array $orders): array
	{
		$result = [
			'ok'    => 0,
			'error' => 0,
			'files' => [],
		];

		foreach ($orders as $order) {
			$file = MYGLS_DIR . '/labels/' . $order->getOrder()->getId() . '.pdf';

			if (file_exists($file)) {
				$result['files'][] = $file;
				$result['ok']++;
			}
		}

		return $result;
	}

	public function checkCompleted(GlsOrder $glsOrder): bool
	{
		$api = new Api;

		try {
			$result = $api->getParcelStatusList($glsOrder->numberPackage);

			if ($result === false) {
				$glsOrder->lastStatus = GlsOrder::STATUS_NOT_FOUND;
				$this->em->persist($glsOrder);
				$this->em->flush($glsOrder);

				return false;
			}

			foreach ($result as $status) {
				if ($status['StCode'] == '23') {
					$glsOrder->lastStatus = GlsOrder::STATUS_RETURNED;
					$this->em->persist($glsOrder);
					$this->em->flush($glsOrder);

					return false;
				}
			}

			if (isset($result[0]['StCode']) && $result[0]['StCode'] == '5') {
				return true;
			}
		} catch (Exception $e) {
			Debugger::log("Cannot check complete data for order '{$glsOrder->getOrder()->getId()}' ({$glsOrder->numberPackage}) - "
				. $api->getTrackingUrlXml($glsOrder->numberPackage), 'myGls');
		}

		return false;
	}
}
