<?php declare(strict_types = 1);

namespace MyGls\Model;

use Core\Model\Helpers\BaseEntityService;
use MyGls\Model\Entities\GlsOrder;

/**
 * @method GlsOrder|null getReference($id)
 * @method GlsOrder[] getAll()
 * @method GlsOrder|null get($id)
 */
class OrdersExported extends BaseEntityService
{
	protected $entityClass = GlsOrder::class;

	/**
	 * @param int[]|string[] $ids
	 *
	 * @return GlsOrder[]
	 */
	public function getOrdersNotExported(array $ids = []): array
	{
		return $this->getOrders(false, $ids);
	}

	/**
	 * @param int[]|string[] $ids
	 *
	 * @return GlsOrder[]
	 */
	public function getOrdersExported(array $ids = [], bool $loadFull = true): array
	{
		return $this->getOrders(true, $ids, $loadFull);
	}

	public function getOrdersNotCompleted(): array
	{
		$data = [];

		foreach ($this->getEr()->createQueryBuilder('go')
			         ->where('go.lastStatus NOT IN (:status) OR go.lastStatus IS NULL')
			         ->andWhere('go.numberPackage IS NOT NULL')
			         ->setParameters([
				         'status' => [
					         GlsOrder::STATUS_COMPLETED,
					         GlsOrder::STATUS_NOT_FOUND,
					         GlsOrder::STATUS_RETURNED,
				         ],
			         ])->getQuery()
			         ->getResult() as $row) {
			$data[$row->getOrder()->getId()] = $row;
		}

		return $data;
	}

	/**
	 * @param int[]|string[] $ids
	 *
	 * @return GlsOrder[]
	 */
	public function getOrders(?bool $isExported = null, array $ids = [], bool $loadFull = true): array
	{
		$qb = $this->getEr()->createQueryBuilder('go')
			->addSelect('anp')
			->leftJoin('go.associatedNumberPackages', 'anp')
			->orderBy('go.order')->groupBy('go.order');

		if ($loadFull) {
			$qb->addSelect('o, go, oad, oai, os, s, oCurrency')
				->join('go.order', 'o')
				->leftJoin('o.currency', 'oCurrency')
				->leftJoin('o.addressDelivery', 'oad')
				->leftJoin('o.addressInvoice', 'oai')
				->leftJoin('o.spedition', 'os')
				->leftJoin('os.spedition', 's');
		}

		if ($isExported === true) {
			$qb->where('go.exported IS NOT NULL');
		} else if ($isExported === false) {
			$qb->where('go.exported IS NULL');
		}

		if (!empty($ids)) {
			$qb->andWhere('go.order IN (:ids)')->setParameter('ids', $ids);
		}

		$result = [];
		foreach ($qb->getQuery()->getResult() as $item) {
			$result[$item->getOrder()->getId()] = $item;
		}

		return $result;
	}

	public function markAsCompleted(array $orderIds): void
	{
		if (!empty($orderIds)) {
			$this->em->getConnection()->executeStatement("UPDATE my_gls__order SET last_status = '" . GlsOrder::STATUS_COMPLETED . "' WHERE order_id IN (" . implode(',', $orderIds) . ")");
		}
	}

	public function findIdByPackageNumber(string $packageNumber): array
	{
		$ids = [];
		foreach ($this->getEr()->createQueryBuilder('g')
			         ->select('IDENTITY(g.order) as id')
			         ->leftJoin('g.associatedNumberPackages', 'acp')
			         ->orWhere('g.numberPackage LIKE :number')
			         ->orWhere('acp.numberPackage LIKE :number')
			         ->setParameter('number', "%$packageNumber%")
			         ->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		return $ids;
	}
}
