<?php declare(strict_types = 1);

namespace MyGls\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Countries;
use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\FrontModule\Model\CartFacade;
use EshopOrders\FrontModule\Model\Dao\PaymentSpedition;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\SetOrderFormDataEvent;
use EshopOrders\Model\PaymentSpeditions;
use MyGls\Model\Entities\GlsOrder;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderFormSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected PaymentSpeditions      $paymentSpeditions;
	protected Translator             $translator;
	protected Request                $httpRequest;
	protected Countries              $countries;
	protected CartFacade             $cartFacade;

	public function __construct(
		EntityManagerDecorator $em,
		PaymentSpeditions $paymentSpeditions,
		Translator $translator,
		Request                $request,
		Countries $countries,
		CartFacade             $cartFacade
	)
	{
		$this->em                = $em;
		$this->paymentSpeditions = $paymentSpeditions;
		$this->translator        = $translator;
		$this->httpRequest       = $request;
		$this->countries         = $countries;
		$this->cartFacade        = $cartFacade;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.createOrderForm'        => ['createOrderForm', 100],
			'eshopOrders.setOrderFormData'       => ['setOrderFormData', 100],
			'eshopOrders.orderFormValidate'      => ['orderFormValidate', 100],
			'eshopOrders.orderBeforeSave'        => ['orderBeforeSave', 100],
			'eshopOrders.saveOrderFormDataStep2' => ['saveOrderFormDataStep2', 100],
			'eshopOrders.saveOrderFormDataStep3' => ['saveOrderFormDataStep3', 100],
		];
	}

	public function createOrderForm(OrderFormEvent $event): void
	{
		$form = $event->form;

		$cart = $this->cartFacade->getCart();
		$templates   = $form->getCustomData('speditionServiceTemplates') ?: [];
		$templates[] = __DIR__ . '/speditionTemplate.latte';
		$form->addCustomData('speditionServiceTemplates', $templates);

		$container = $form->addContainer('myGls');

		foreach ($this->getSpeditions() as $countryId => $rows) {
			foreach ($rows as $row) {
				if (!$row->getSpedition()->isPickup) {
					continue;
				}
				$speditionId = $row->getSpedition()->getId();
				$country     = $this->countries->get($countryId);

				$k = $country->getId() . '_' . $speditionId;
				if ($container->getComponent($k, false)) {
					continue;
				}

				$container->addHidden($k)
					->getControlPrototype()->addCLass('myGlsTarget_' . $k);
				foreach (['street', 'city', 'postal'] as $col) {
					$container->addHidden($k . '_' . $col);
				}
				$pointInput                          = $container->addText($k . '_p');
				$pointInput->setHtmlAttribute('readonly')
					->setHtmlAttribute('data-country', $country->getId())
					->setHtmlAttribute('data-mygls-pickup-trigger', $container->getComponent($k)->getHtmlId())
					->setPlaceholder('myGlsFront.selectDispensingPoint')
					->getControlPrototype()->addClass('myGlsTrigger_' . $k);

				if ($cart->hasDisabledDeliveryBoxes()) {
					$pointInput->setHtmlAttribute('data-mygls-disabled', '17,18'); // boxy
					// https://e-balik.cz/images2/document/Implementace_PSD.pdf
				}

				foreach (['street', 'city', 'postal'] as $col) {
					$pointInput->setHtmlAttribute('data-' . $col, $container->getComponent($k . '_' . $col)->getHtmlId());
				}

				$event->template->myGlsLastKey = $k;
			}
		}
	}

	public function setOrderFormData(SetOrderFormDataEvent $event): void
	{
		$data = $event->data;
		$form = $event->form;

		foreach ($data['myGls'] ?? [] as $k => $v) {
			if (isset($form->getComponent('myGls', false)[$k])) {
				$form->getComponent('myGls')[$k]->setDefaultValue($v);
			}
		}
	}

	public function saveOrderFormDataStep2(SaveOrderFormDataEvent $event): void
	{
		$data    = &$event->data;
		$request = $this->httpRequest;
		$post    = $request->getPost();

		$spedition = $this->getSpeditions()[$data['speditionCountry']][$data['payment'] . '_' . $data['spedition']] ?? null;

		if ($spedition && $spedition->getSpedition()->isPickup) {
			$k = $data['speditionCountry'] . '_' . $data['spedition'];

			$data['myGls'][$k] = $request->getPost('myGls')[$k];
			foreach (['p', 'street', 'city', 'postal'] as $v) {
				$data['myGls'][$k . '_' . $v] = $request->getPost('myGls')[$k . '_' . $v];
			}

			$data['disableDeliveryAddressSpedition'] = 'myGls';
			$data['disableDeliveryAddress']          = true;
			$data['useAddrDeli']                     = false;

			foreach (['street2', 'city2', 'postal2'] as $col) {
				if (isset($post[$col])) {
					$data[$col] = $post[$col];
				}
			}
		} else if ($data['disableDeliveryAddressSpedition'] == 'myGls') {
			unset($data['disableDeliveryAddress']);
			unset($data['useAddrDeli']);
		}
	}

	public function saveOrderFormDataStep3(SaveOrderFormDataEvent $event): void
	{
		$data = &$event->data;
		$form = &$event->form;

		$spedition = $this->getSpeditions()[$data['speditionCountry']][$data['payment'] . '_' . $data['spedition']] ?? null;

		if ($spedition && $spedition->getSpedition()->isPickup) {
			$k = $data['speditionCountry'] . '_' . $data['spedition'];
			$d = [];
			foreach (['company', 'firstName', 'lastName', 'email', 'phone'] as $v) {
				$d[$v . '2'] = $data[$v];
			}
			$countryId   = $data['speditionCountry'];
			$countryText = $this->countries->getAllNameColumn()[$countryId] ?? $countryId;
			$d           += [
				'useAddrDeli'  => true,
				'street2'      => $data['myGls'][$k . '_street'],
				'city2'        => $data['myGls'][$k . '_city'],
				'postal2'      => $data['myGls'][$k . '_postal'],
				'country2'     => $countryId,
				'country2Text' => $countryText,
			];

			$data = array_merge($data, $d);
			$form->setValues($d);
		}
	}

	public function orderFormValidate(OrderFormEvent $event): void
	{
		$form   = $event->form;
		$values = $form->getValues();

		$spedition = $this->getSpeditions()[$values->speditionCountry][$values->payments->{$values->speditionCountry} . '_' . $values->speditions->{$values->speditionCountry}] ?? null;
		if ($spedition && $spedition->getSpedition()->isPickup) {
			$branchId = $values->myGls->{$values->speditionCountry . '_' . $spedition->getSpedition()->getId()};

			if (!$branchId) {
				$form->addError($this->translator->translate('myGls.orderForm.myGlsPointMissing'));
			}
		}
	}

	public function orderBeforeSave(OrderEvent $event): void
	{
		/** @var ArrayHash $values */
		$values = $event->formData;

		$spedition = $this->getSpeditions()[$values->speditionCountry][$values->payments->{$values->speditionCountry} . '_' . $values->speditions->{$values->speditionCountry}] ?? null;
		if ($spedition) {
			$entity = new GlsOrder($event->order);

			if ($spedition->getSpedition()->isPickup) {
				$k                    = $values->speditionCountry . '_' . $spedition->getSpedition()->getId();
				$entity->parcelNumber = $values->myGls->$k;

				$orderSpedition = $event->order->getSpedition();
				if ($orderSpedition) {
					$orderSpedition->deliveryPointId = (string) $values->myGls->$k;
					$this->em->persist($orderSpedition);
				}
			} else {
				$orderSpedition = $event->order->getSpedition();
				if ($orderSpedition) {
					$orderSpedition->deliveryPointId = null;
					$this->em->persist($orderSpedition);
				}
			}

			$this->em->persist($entity);
		}
	}

	/**
	 * @return PaymentSpedition[][]
	 */
	protected function getSpeditions(): array
	{
		$result = [];
		foreach ($this->paymentSpeditions->getAllPublished() as $row) {
			if ($row->getSpedition()->getIdent() !== 'myGls') {
				continue;
			}

			foreach ($row->getCountries() as $country) {
				$result[$country->getId()][$row->getPayment()->getId() . '_' . $row->getSpedition()->getId()] = $row;
			}
		}

		return $result;
	}
}
