<?php declare(strict_types = 1);

namespace Navigations\Model\Entities;

use Core\Model\Entities\TTranslateListener;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;
use Nette\Utils\DateTime;

/**
 * Class Navigation
 * @package Navigations\Model\Entities
 * @Gedmo\Tree(type="nested")
 * @ORM\Table(name="navigations__navigation", indexes={@ORM\Index(name="type_idx", columns={"type"})})
 * @ORM\Entity(repositoryClass="Core\Model\Entities\Repository\NestedTreeRepository")
 * @ORM\EntityListeners({"NavigationListener", "Core\Model\Entities\TranslateListener"})
 */
class Navigation
{
	use Identifier;
	use MagicAccessors;
	use TTranslateListener;

	/**
	 * @var NavigationText[]
	 * @ORM\OneToMany(targetEntity="NavigationText", mappedBy="navigation", indexBy="lang")
	 */
	protected $texts;

	/**
	 * @var string
	 * @ORM\Column(name="link", type="string", nullable=false, length=255)
	 */
	public $link;

	/**
	 * @var string
	 * @ORM\Column(name="type", type="string", nullable=false)
	 */
	public $componentType;

	/**
	 * @var array
	 * @ORM\Column(name="component_params", type="array", nullable=true)
	 */
	public $componentParams;

	/**
	 * @var array
	 * @ORM\Column(name="attribs", type="array", nullable=true)
	 */
	private $params;

	/**
	 * @var NavigationGroup
	 * @ORM\ManyToOne(targetEntity="NavigationGroup")
	 * @ORM\JoinColumn(name="group_id", referencedColumnName="id", nullable=false, onDelete="RESTRICT")
	 */
	private $group;

	/**
	 * @var Navigation[]
	 * @ORM\OneToMany(targetEntity="Navigation", mappedBy="parent")
	 * @ORM\OrderBy({"lft" = "ASC"})
	 */
	private $children;

	/**
	 * @var Navigation
	 * @Gedmo\TreeParent
	 * @ORM\ManyToOne(targetEntity="Navigation", inversedBy="childrens")
	 * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", onDelete="RESTRICT")
	 */
	private $parent;

	/**
	 * @Gedmo\TreeLeft
	 * @ORM\Column(name="lft", type="integer")
	 */
	private $lft;

	/**
	 * @Gedmo\TreeLevel
	 * @ORM\Column(name="lvl", type="integer")
	 */
	private $lvl;

	/**
	 * @Gedmo\TreeRight
	 * @ORM\Column(name="rgt", type="integer")
	 */
	private $rgt;

	/**
	 * @var Navigation
	 * @Gedmo\TreeRoot
	 * @ORM\ManyToOne(targetEntity="Navigation")
	 * @ORM\JoinColumn(name="tree_root", referencedColumnName="id", nullable=true, onDelete="CASCADE")
	 */
	private $root;

	/**
	 * @var DateTime
	 * @Gedmo\Timestampable(on="update")
	 * @ORM\Column(name="modified", type="datetime", nullable=true)
	 */
	private $modified;

	/**
	 * @var DateTime
	 * @Gedmo\Timestampable(on="create")
	 * @ORM\Column(name="created", type="datetime", nullable=true)
	 */
	private $created;

	public function __construct(NavigationGroup $group, string $type)
	{
		$this->group         = $group;
		$this->componentType = $type;
		$this->children      = new ArrayCollection();
		$this->texts         = new ArrayCollection();
		$this->link          = '';
	}

	public function getParams(): array { return $this->params ?: []; }

	public function getParam(string $key) { return $this->params[$key] ?? null; }

	public function setParam(string $key, $value): self
	{
		if ($value) {
			$this->params[$key] = $value;
		} else {
			unset($this->params[$key]);
		}

		return $this;
	}

	public function setTexts(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	public function getText(?string $lang = null): ?NavigationText { return $this->texts[$lang ?: $this->locale] ?? null; }

	/** @return NavigationText[]|ArrayCollection */
	public function getTexts() { return $this->texts; }

	public function getGroup(): NavigationGroup { return $this->group; }

	public function setGroup(NavigationGroup $group): self
	{
		$this->group = $group;

		return $this;
	}

	public function getParent(): ?Navigation { return $this->parent; }

	public function setParent(?Navigation $parent = null): self
	{
		$this->parent = $parent;

		return $this;
	}

	/** @return Navigation[] */
	public function getChildren() { return $this->children; }

	public function getRoot(): Navigation { return $this->root; }

	public function getLft(): int { return $this->lft; }

	public function getRgt(): int { return $this->rgt; }

	public function getLvl(): int { return $this->lvl; }

	public function getModified(): DateTime { return $this->modified; }

	public function getCreated(): DateTime { return $this->created; }
}
