<?php declare(strict_types = 1);

namespace Navigations\Model;

use Contributte\EventDispatcher\EventDispatcher;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Helpers\BaseEntityService;
use Core\FrontModule\Model\Redirects;
use Navigations\Model\Entities\NavigationGroup;
use Navigations\Model\Entities\NavigationGroupText;
use Navigations\Model\Entities\NavigationText;
use Navigations\Model\Event\RouteInEvent;
use Gedmo\Tree\Entity\Repository\NestedTreeRepository;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Helper\NavigationsHelper;
use Navigations\Model\Providers\INavigationItem;
use Nette\Caching\Cache;
use Nette\Localization\ITranslator;

/**
 * Class Navigations
 * @package Navigations\Model
 *
 * @method Navigation|object|null getReference($id)
 * @method Navigation|null get($id)
 * @method NestedTreeRepository getEr()
 */
class Navigations extends BaseEntityService
{
	/** @var string */
	const CACHE_NAMESPACE = 'navigations';

	/** @var string */
	protected $entityClass = Navigation::class;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => [self::CACHE_NAMESPACE],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	/** @var DaoNavigationItem[][] */
	public $cNavs;

	/** @var DaoNavigationItem[][] */
	public $cNavsTree;

	/** @var ITranslator @inject */
	public $translator;

	/** @var ItemsCollector @inject */
	public $itemsCollector;

	/** @var Redirects @inject */
	public $redirectsService;

	/** @var EventDispatcher @inject */
	public $eventDispatcher;

	/** @var NavigationsHelper @inject */
	public $navigationsHelper;

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	/**
	 * @param int    $id
	 * @param string $lang
	 *
	 * @return DaoNavigationItem[]
	 */
	public function getPath(int $id, ?string $lang = null): array
	{
		$lang = $lang ?: $this->translator->getLocale();
		$key  = 'path/' . $id . '/' . $lang;

		return $this->getCache()->load($key, function(&$dep) use ($id, $lang) {
			$dep = $this->cacheDep;

			$result = [];
			$stmt   = $this->em->getConnection()->prepare("
						SELECT T2.id 
						FROM (
							SELECT @r AS _id,
								(SELECT @r := parent_id FROM navigations__navigation WHERE id = _id) AS parent_id, @l := @l + 1 AS lvl 
								FROM (SELECT @r := {$id}, @l := 0) vars, navigations__navigation h WHERE @r <> 0) T1 
						JOIN navigations__navigation T2 ON T1._id = T2.id ORDER BY T1.lvl DESC");
			$stmt->execute();

			foreach ($stmt->fetchAll() as $row) {
				$nav = $this->getNavigation((int) $row['id'], $lang);
				if ($nav)
					$result[$row['id']] = $nav;
			}

			return $result;
		});
	}

	/**
	 * @param string $lang
	 *
	 * @return DaoNavigationItem[]
	 */
	public function getPublishedToDaoNavigationItem(?string $lang = null): array
	{
		$lang = $lang ?: $this->translator->getLocale();

		if (!isset($this->cNavs[$lang])) {
			$key = 'publishedNavs/' . $lang;

			$this->cNavs[$lang] = $this->getCache()->load($key, function(&$dep) use ($lang) {
				$dep    = $this->cacheDep;
				$result = [];

				$loop = function(array $arr) use (&$loop, &$result) {
					/** @var DaoNavigationItem[] $arr */
					foreach ($arr as $v) {
						$result[$v->id] = $v;

						if ($v->childs)
							$loop($v->childs);
					}
				};

				foreach ($this->getPublishedDaoNavigationStructure($lang) as $v)
					$loop($v);

				return $result;
			});
		}

		return $this->cNavs[$lang];
	}

	public function getPublishedDaoNavigationStructure(?string $lang = null): array
	{
		$lang = $lang ?: $this->translator->getLocale();

		if (!isset($this->cNavsTree[$lang])) {
			$key = 'publishedStructure/' . $lang;

			$this->cNavsTree[$lang] = $this->getCache()->load($key, function(&$dep) use ($lang) {
				$dep = $this->cacheDep;
				/** @var DaoNavigationItem[] $flat */
				$flat    = [];
				$grouped = [];
				$tree    = [];

				foreach ($this->navigationsHelper->getPublished($lang) as $nav) {
					$grouped[$nav->groupType][$nav->parentId][$nav->id] = $nav;
					$flat[$nav->id]                                     = &$grouped[$nav->groupType][$nav->parentId][$nav->id];
				}
				$this->em->clear([
					Navigation::class,
					NavigationText::class,
					NavigationGroup::class,
					NavigationGroupText::class,
				]);

				$createTree = function(&$list, $parent) use (&$createTree) {
					$tree = [];
					foreach ($parent as $k => $l) {
						if (isset($list[$l->id])) {
							if ($l->lvl >= 1)
								foreach ($list[$l->id] as $kk => $v) {
									$list[$l->id][$kk]->parent = &$parent[$k];
								}
							$l->isParent = true;
							$l->childs   = $createTree($list, $list[$l->id]);
						}

						$tree[] = $l;
					}

					return $tree;
				};

				foreach ($this->getEr()->createQueryBuilder('n')->select('n.id, ng.type')
					         ->join('n.group', 'ng')
					         ->where('n.lvl = 0')->getQuery()->getScalarResult() as $row) {
					if (!isset($grouped[$row['type']]))
						continue;

					$dao                        = new DaoNavigationItem();
					$dao->id                    = $row['id'];
					$dao->parent                = 0;
					$grouped[$row['type']][0][] = $dao;

					$tree[$row['type']] = $createTree($grouped[$row['type']], $grouped[$row['type']][0])[0]->childs;
				}

				$aliases = [];

				foreach ($flat as $k => $v) {
					if ($v->componentType == 'navigation.alias') {
						$flat[$k]->setParam('aliasReference', $flat[$v->componentParams['navigation']]);
						$flat[$k]->link = $flat[$v->componentParams['navigation']]->link;
						$aliases[$k]    = $v;
						continue;
					}

					$flat[$k]->link = $this->generateLink($v);
				}

				foreach ($aliases as $k => $v) {
					$flat[$k]->link = $this->generateLink($v);
				}

				return $tree;
			});
		}

		return $this->cNavsTree[$lang];
	}

	/**
	 * @param string $component
	 *
	 * @return DaoNavigationItem[]
	 */
	public function getPublishedByComponent(string $component, ?string $lang = null): array
	{
		$result = [];
		foreach ($this->getPublishedToDaoNavigationItem($lang) as $nav) {
			if ($nav->componentType === $component)
				$result[$nav->id] = $nav;
		}

		return $result;
	}

	public function getNavigation(int $id, ?string $lang = null): ?DaoNavigationItem
	{
		foreach ($this->getPublishedToDaoNavigationItem($lang) as $nav) {
			if ($nav->id === $id)
				return $nav;
		}

		return null;
	}

	public function generateLink(DaoNavigationItem $navigation): ?string
	{
		$t = [
			'locale'           => $navigation->lang,
			'activeNavigation' => $navigation,
		];

		return $this->getUrlById($t);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function getHomepage(?string $lang = null): ?DaoNavigationItem
	{
		foreach ($this->getPublishedToDaoNavigationItem($lang) as $v) {
			if ($v->isHomepage)
				return $v;
		}

		return null;
	}

	public function getHomepageForRouter(?string $lang = null): array
	{
		$hp = $this->getHomepage($lang);
		if ($hp) {
			$component = $this->itemsCollector->getItemById($hp->componentType);

			return $component->routerIn($hp, ['getHomepage' => 1]) + ['activeNavigation' => $hp];
		}

		return [];
	}

	public function getIdByUrl($url): ?array
	{
		try {
			$navigation               = null;
			$locale                   = $url['locale'] ?? $this->translator->getLocale();
			$navigations              = $this->getPublishedToDaoNavigationItem($locale);
			$navigationsByLink        = [];
			$implodedPartsForRedirect = [];
			$pathParts                = explode('/', $url['path']);
			$tmpPath                  = $url['path'];
			$tmpPathParts             = $pathParts;
			$return                   = [];
			$i                        = 0;

			if ($this->translator->getDefaultLocale() !== $locale)
				$tmpPath = $locale . '/' . $tmpPath;

			foreach ($navigations as $k => $nav) {
				if ($nav->componentType === 'navigation.alias' || !$nav->link)
					continue;

				$navigationsByLink[ltrim($nav->link, '/')] = $k;
			}

			// Vyhledání navigace která odpovídá url. Obsah za lomítkem se postupně odebírá
			while (!$navigation && $i < 15 && !empty($tmpPathParts)) {
				$implodedPartsForRedirect[] = $tmpPath;
				$i++;

				foreach ($navigationsByLink as $link => $k) {
					if ($link == $tmpPath) {
						$navigation = $navigations[$k];
						break 2;
					}
				}

				array_pop($tmpPathParts);
				$tmpPath = implode('/', $tmpPathParts);
			}

			// Pokud se navigace nenalezne, tak se podíváme do přesměrání a případně nastavíme navigace nebo rovnou přesměrujeme
			if (!$navigation) {
				$redirects = $this->redirectsService->getEr()->createQueryBuilder('r')
					->select('r.to, r.package, r.relationKey, r.relationValue')
					->where('r.from IN (:from)')
					->setParameter('from', $implodedPartsForRedirect)
					->setMaxResults(1)->getQuery()->getArrayResult();

				if (isset($redirects[0])) {
					$redirects = $redirects[0];

					if ($redirects['relationValue']) {
						$navigation = $navigations[$redirects['relationValue']];
					} else {
						header('Location: ' . $this->navigationsHelper->getHttpRequest()->getUrl()->getBaseUrl() . ltrim($redirects['to'], '/'));
					}
				}
			}

			// Pokud navigace existuje, tak vytvoříme data, jinak zavoláme událost
			if ($navigation) {
				$out = $this->getIdByUrlRouterIn($navigation, $url);

				if (is_array($out))
					$return += $out + ['activeNavigation' => $navigation];
			} else {
				$this->eventDispatcher->dispatch(Navigations::class . '::routeInNotFound', new RouteInEvent(null, $url, $return));
			}

			// Pokud data neexistujou, tak zkusíme vytáhnout data z homepage
			if (empty($return)) {
				$hp = $this->getHomepage($url['locale']);

				if ($hp) {
					$out = $this->getIdByUrlRouterIn($hp, $url);

					if (is_array($out))
						$return += $out + ['activeNavigation' => $hp];
				}
			}

			// Pokud data stále nejsou tak stránka neexistuje
			if (empty($return))
				return null;

			// Vyparsujeme paginator
			$returnPaginator = [];
			$paginatorIndex  = array_search($this->translator->translate('default.urlPart.page'), $pathParts);
			if ($paginatorIndex !== false) {
				$returnPaginator['list-page'] = (int) $pathParts[$paginatorIndex + 1] ?? 1;
				$returnPaginator['do']        = 'list-paginator';
			}

			return $return + $returnPaginator;
		} catch (\Exception $e) {
			return null;
		}
	}

	protected function getIdByUrlRouterIn(DaoNavigationItem $navigation, array $params)
	{
		$component = $this->itemsCollector->getItemById($navigation->componentType);

		return $component ? $component->routerIn($navigation, $params) : null;
	}

	public function getUrlById(&$params)
	{
		$return = null;
		try {
			$tmp = $params;
			if ($tmp['activeNavigation']) {
				$tmp['activeNavigation'] = $tmp['activeNavigation']->id;
			}

			foreach ($tmp as $v)
				if (is_object($v))
					unset($tmp[$v]);

			$key = 'urlById/' . serialize($tmp);

			$data = $this->getCache()->load($key, function(&$dep) use ($params, $return) {
				$dep = $this->cacheDep;

				if (isset($params['activeNavigation'])) {
					$navType = $params['activeNavigation']->componentType;

					if ($navType == 'navigation.alias') {
						$tmp = $params['activeNavigation']->getParam('aliasReference');

						if ($tmp) {
							$params['activeNavigation'] = $tmp;
							$navType                    = $tmp->componentType;
						}
					}
					$component = $this->itemsCollector->getItemById($navType);

					if ($component)
						$return = $this->getPreAlias($params['activeNavigation']) . $component->routerOut($params['activeNavigation'], $params);
				}

				if (!$return) {
					$groups = $this->itemsCollector->findItemByPresenter($params['presenter'], $params['action']);

					if (!$groups) {
						if ($params['presenter']) {
							preg_match('/Override$/', $params['presenter'], $matches, PREG_OFFSET_CAPTURE, 0);

							if (isset($matches[0]))
								$params['presenter'] = substr($params['presenter'], 0, $matches[0][1]);
						}

						$groups = $this->itemsCollector->findItemByPresenter($params['presenter'], $params['action']);
					}

					if (!$groups)
						return null;

					ksort($groups);

					foreach ($groups as $components) {
						/** @var INavigationItem[]|BaseItem[] $components */
						foreach ($components as $component) {
							foreach ($this->getPublishedByComponent($component->getId(), $params['locale']) as $navigation) {
								$return = $component->routerOut($navigation, $params);

								if ($return === '')
									$return = null;
								else if ($return !== null && $return !== false) {
									$component->updateCacheDep($dep, $params);
									$return = $this->getPreAlias($navigation) . $return;
									break 3;
								}
							}
						}
					}
				}

				// Upravení url pro stránkování
				if (isset($params['do']) && isset($params['list-page']) && $params['do'] == 'list-paginator') {
					if ($params['list-page'] > 1) {
						$return .= '/' . $this->translator->translate('default.urlPart.page') . '/' . $params['list-page'];
					}

					unset($params['list-page']);
					unset($params['do']);
				}

				if (isset($params['activeNavigation']) && !$params['activeNavigation']->link)
					$params['activeNavigation']->link = $return;

				return ['return' => $return, 'params' => $params];
			});

			$params = $data['params'];
			$return = $data['return'];

			return $return;
		} catch (\Exception $e) {
		}

		return null;
	}

	/**
	 * @param DaoNavigationItem $navigation
	 *
	 * @return string
	 */
	public function getPreAlias(DaoNavigationItem $navigation)
	{
		$arr = [];
		$t   = $navigation->parent;

		while ($t) {
			$arr[] = $t->alias;
			$t     = $t->parent;
		}

		$url = $arr ? '/' . implode('/', array_reverse($arr)) : '';
		$this->setLocaleUrlPrefix($url, $navigation);

		return $url;
	}

	public function setLocaleUrlPrefix(string &$url, ?DaoNavigationItem $navigation = null): void
	{
		if ($navigation->getLang() && $this->translator->getDefaultLocale() != $navigation->getLang())
			$url = '/' . $navigation->getLang() . $url;
	}

	/*******************************************************************************************************************
	 * ===========================  Cache
	 */
}
