<?php declare(strict_types = 1);

namespace Navigations\Model\Entities;

use Core\Model\Entities\TLang;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Kdyby\Doctrine\Entities\Attributes\Identifier;
use Kdyby\Doctrine\Entities\MagicAccessors;
use Nette\Utils\Strings;

/**
 * @ORM\Table(name="navigations__navigation", indexes={@ORM\Index(name="lang_idx", columns={"lang"})})
 * @ORM\Entity
 * @ORM\EntityListeners({"NavigationListener"})
 */
class Navigation
{
	use Identifier;
	use MagicAccessors;
	use TLang;

	/**
	 * @var string
	 * @ORM\Column(name="title", type="string", length=255, nullable=false)
	 */
	public $title;

	/**
	 * @var string
	 * @ORM\Column(name="alias", type="string", length=255, nullable=false)
	 */
	public $alias;

	/**
	 * @var string
	 * @ORM\Column(name="meta_description", type="string", length=255, nullable=true)
	 */
	public $metaDescription;

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint", nullable=false)
	 */
	public $isPublished;

	/**
	 * @var int
	 * @ORM\Column(name="is_homepage", type="smallint", nullable=false, options={"default"=0})
	 */
	public $isHomepage;

	/**
	 * @var string
	 * @ORM\Column(name="link", type="string", nullable=false)
	 */
	public $link;

	/**
	 * @var string
	 * @ORM\Column(name="type", type="string", nullable=false)
	 */
	public $componentType;

	/**
	 * @var array
	 * @ORM\Column(name="content_params", type="array", nullable=true)
	 */
	public $componentParams;

	/**
	 * @var array
	 * @ORM\Column(name="image", type="string", nullable=true)
	 */
	public $image;

	/**
	 * @var array
	 * @ORM\Column(name="attribs", type="array", nullable=true)
	 */
	private $params;

	/**
	 * @var string
	 * @ORM\Column(name="background", type="string", length=255, nullable=true)
	 */
	protected $background;

	/**
	 * @var NavigationGroup
	 * @Gedmo\SortableGroup
	 * @ORM\ManyToOne(targetEntity="NavigationGroup")
	 * @ORM\JoinColumn(name="group_id", referencedColumnName="id", nullable=false, onDelete="RESTRICT")
	 */
	public $group;

	/**
	 * @var int
	 *
	 * @Gedmo\SortablePosition
	 * @ORM\Column(name="position", type="integer")
	 */
	private $position;

	/**
	 * @var Navigation[]
	 * @ORM\OneToMany(targetEntity="Navigation", mappedBy="parent")
	 * @ORM\OrderBy({"position" = "ASC"})
	 */
	private $children;

	/**
	 * @var Navigation
	 * @ORM\ManyToOne(targetEntity="Navigation", inversedBy="childrens")
	 * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $parent;

	public function __construct($title, $group, $link, $type, $isPublished = 1)
	{
		$this->title         = $title;
		$this->group         = $group;
		$this->link          = $link;
		$this->componentType = $type;
		$this->isPublished   = $isPublished;
		$this->isHomepage    = 0;
		$this->children      = new ArrayCollection();
		$this->setAlias($title);
		$this->setPosition(-1);
	}

	public function setAlias($alias, $lang = null)
	{
		if ($alias)
			$this->alias = $lang && in_array($lang, ['he']) ? $alias : Strings::webalize($alias);
	}

	/******
	 * === Position
	 */

	public function setPosition($position)
	{
		$this->position = $position;
	}

	public function getPosition()
	{
		return $this->position;
	}

	/******
	 * === Params
	 */

	public function getParams() { return $this->params ?: []; }

	public function getParam($key) { return $this->params[$key] ?? null; }

	public function setParam($key, $value)
	{
		if ($value) {
			$this->params[$key] = $value;
		} else {
			unset($this->params[$key]);
		}
	}

	/******
	 * === Parent
	 */

	public function setParent($parent = null)
	{
		$this->parent = $parent;
	}

	public function getParent() { return $this->parent; }

	public function getChildren() { return $this->children; }
}
