<?php declare(strict_types = 1);

namespace Navigations\AdminModule\Components;

use Core\FrontModule\Model\SeoContainer;
use Core\Model\Helpers\Arrays;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\CheckboxListInput;
use Core\Model\UI\Form\Controls\SelectInput;
use Navigations\Model\BaseItem;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\ItemsCollector;
use Navigations\AdminModule\Model\Navigations;
use Navigations\Model\NavigationsGroups;
use Nette\Forms\Form;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;
use Nette\Utils\Strings;

/**
 * TODO předělat načítání componentType, aby bylo bez ajaxu jako je u TemplatePages
 *
 * Class NavigationForm
 * @package Navigations\AdminModule\Components
 */
class NavigationForm extends BaseControl
{
	/** @var Navigation */
	public $navigation;

	/** @var ItemsCollector */
	protected $itemsCollector;

	/** @var NavigationsGroups */
	protected $navigationsGroups;

	/** @var Navigations */
	protected $navigationsService;

	/** @var SeoContainer */
	protected $seoContainerService;

	public function __construct(ItemsCollector $itemsCollector, NavigationsGroups $navigationsGroups, Navigations $navigations,
	                            SeoContainer $seoContainer)
	{
		$this->itemsCollector      = $itemsCollector;
		$this->navigationsGroups   = $navigationsGroups;
		$this->navigationsService  = $navigations;
		$this->seoContainerService = $seoContainer;
	}

	public function render()
	{
		$this->template->thisForm = $this['form'];
		$this->template->setFile(__DIR__ . '/NavigationForm.latte');
		$this->template->render();
	}

	protected function attached($presenter)
	{
		parent::attached($presenter);

		$componentType = $this->getPresenter()->getHttpRequest()->getPost('componentType');
		if ($componentType)
			$this->loadComponent($componentType);
	}

	protected function loadComponent($componentId)
	{
		$form = $this['form'];

		if ($form->getComponent('component', false)) {
			$form->removeComponent($form['component']);
		}

		$component = $this->itemsCollector->getItemById($componentId);

		if ($component) {
			$form->addComponent($component->getFormContainer(), 'component');
			$component = $form['component'];
			if ($this->navigation) {
				foreach ($component->components as $k => $c) {
					$value = $this->navigation->componentParams[$k] ?? null;

					if (!$value)
						continue;

					// TODO dát někam jinam. To samé se používá u TemplateReader
					if ($c instanceof SelectInput) {
						if (array_key_exists($value, $c->getItems())) {
							$c->setDefaultValue($value);
						}
					} else if ($c instanceof CheckboxListInput) {
						$tmp = [];
						foreach ($value as $v) {
							if (array_key_exists($v, $c->getItems()))
								$tmp[] = $v;
						}

						$c->setDefaultValue($tmp);
					} else {
						$c->setDefaultValue($value);
					}
				}
			}
		}
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleLoadInputs($componentId)
	{
		$presenter = $this->getPresenter();

		try {
			$this->loadComponent($componentId);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (\Exception $e) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('component');
		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();
		//		$form->setAjax();

		$components = ['' => $this->t('default.choose')];
		foreach ($this->itemsCollector->getItems() as $groupName => $group) {
			$arr = [];

			foreach ($group['items'] as $key => $class) {
				/** @var $class BaseItem */
				$arr[$class->getId()] = $this->t($class->getTitle());
			}

			$components[$this->t($group['title'])] = $arr;
		}

		$flat     = [];
		$parentsQ = $this->navigationsService->getEr()->createQueryBuilder('n')->select('n.id, n.title, p.id as parentId')
			->leftJoin('n.parent', 'p')->join('n.group', 'g')->orderBy('g.title', 'ASC')->addOrderBy('n.position', 'ASC');
		if ($this->navigation)
			$parentsQ = $parentsQ->andWhere('n.id != :id')->setParameter('id', $this->navigation->getId());
		foreach ($parentsQ->getQuery()->getArrayResult() as $n) {
			$n['parentId'] = $n['parentId'] ?: 0;
			$flat[]        = $n;
		}
		$tree    = Arrays::buildTree($flat, 'parentId', 'id');
		$parents = ['' => ''] + Arrays::builtSelectOptionsTree($tree);

		$navGroups = [];
		foreach ($this->navigationsGroups->getAll() as $group)
			$navGroups[$group->getId()] = $group->title;

		$openIn = [
			'default' => 'navigations.openIn.default',
			'_blank'  => 'navigations.openIn.blank',
		];

		$form->addText('title', 'navigations.navigation.title')->setMaxLength(255)->setRequired();
		$form->addText('alias', 'navigations.navigation.alias')->setMaxLength(255);
		$form->addBool('isPublished', 'default.isPublished')->setDefaultValue(1);
		$form->addSelect('parent', $this->t('default.parent'), $parents)->setTranslator(null);
		$form->addFilesManager('image', 'default.icon');
		$form->addSelect('openIn', 'navigations.navigation.openIn', $openIn)->setDefaultValue('default');
		$form->addSelect('navigationGroup', 'navigations.navigation.navigationGroup', $navGroups)->setRequired();
		$form->addLangsSelect('lang', 'default.language');
		$form->addText('linkClass', 'navigations.navigation.linkClass');
		$form->addText('pageClass', 'navigations.navigation.pageClass');
		$form->addSelect('componentType', $this->t('navigations.navigation.component'), $components)->setTranslator(null)->setRequired();

		$form->addComponent($this->seoContainerService->getContainer(), 'seo');

		$form->addSaveCancelControl('saveControl');

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values)
	{
		if (!$values->alias)
			$values->alias = Strings::webalize($values->title);

		$lang = $values->lang;

		//TODO přidat podmínku na nadřazenou položku pokud je
		if ((!$this->navigation || $this->navigation->alias != $values->alias)) {
			$nav = $this->navigationsService->findByAlias($values->alias, $values->lang);
			if ($nav && $nav->getLangId() == $lang) {
				$form->addError('navigations.navigationForm.aliasExists');
			}
		}

		if ($form->hasErrors()) {
			if ($form->isAjax()) {
				$this->redrawControl('form');
			} else {
				$this->redirect('this');
			}
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$navGroup   = $this->navigationsGroups->getReference($values->navigationGroup);
			$navigation = $this->navigation ?: new Navigation($values->title, $navGroup, '', $values->componentType, $values->isPublished);

			$navigation->title           = $values->title;
			$navigation->group           = $navGroup;
			$navigation->link            = '';
			$navigation->isPublished     = $values->isPublished;
			$navigation->image           = $values->image;
			$navigation->componentType   = $values->componentType;
			$navigation->componentParams = $this->getPresenter()->getRequest()->getPost('component');

			$navigation->setParam('linkClass', $values->linkClass);
			$navigation->setParam('pageClass', $values->pageClass);
			$navigation->setParam('openIn', $values->openIn == 'default' ? null : $values->openIn);
			$navigation->setParent($values->parent && $values->parent != '' ? $this->navigationsService->getReference($values->parent) : null);

			$navigation->setAlias($values->alias ?: $values->title, $values->lang);
			$navigation->setLang($values->lang);
			$navigation->setSeo($values->seo);

			$flashMessage = $navigation->getId() ? 'navigations.navigationForm.edited' : 'navigations.navigationForm.added';
			$this->em->persist($navigation);
			$this->em->flush();
			$form->addCustomData('navigationId', $navigation->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	/*******************************************************************************************************************
	 * ===========================  Get/Set
	 */

	public function setNavigation($id)
	{
		$this->navigation = $this->navigationsService->get($id);

		if ($this->navigation) {
			$form = $this['form'];
			$n    = $this->navigation;

			$form->setDefaults([
				'title'       => $n->title,
				'alias'       => $n->alias,
				'image'       => $n->image,
				'isPublished' => $n->isPublished,
				'openIn'      => $n->getParam('openIn') ?: 'default',
				'lang'        => $n->getLang(),
			]);

			if ($n->parent && array_key_exists($n->parent->getId(), $form['parent']->getItems()))
				$form['parent']->setDefaultValue($n->parent->getId());

			if (array_key_exists($n->group->getId(), $form['navigationGroup']->getItems()))
				$form['navigationGroup']->setDefaultValue($n->group->getId());

			if (array_key_exists($n->componentType, $form['componentType']->getItems())) {
				$form['componentType']->setDefaultValue($n->componentType)
					->getControlPrototype()->attrs['data-default'] = Json::encode($n->componentParams);
			}

			foreach ($n->getParams() as $k => $v) {
				if (isset($form[$k]))
					$form[$k]->setDefaultValue($v);
			}

			if ($n->componentType) {
				$this->loadComponent($n->componentType);
			}

			$this->seoContainerService->setDefaults($form['seo'], $n->getSeo());
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
