<?php declare(strict_types = 1);

namespace Navigations\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Navigations\AdminModule\Model\Navigations;
use Navigations\Model\NavigationsGroups;

class NavigationsGrid extends BaseControl
{
	/** @var NavigationsGroups */
	protected $navigationsGroupsService;

	/** @var Navigations */
	protected $navigationsService;

	public function __construct(Navigations $navigations, NavigationsGroups $groups)
	{
		$this->navigationsService       = $navigations;
		$this->navigationsGroupsService = $groups;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();

		$qb = $this->navigationsService->getEr()->createQueryBuilder('n')
			->join('n.group', 'g')
			->orderBy('g.id', 'ASC')->addOrderBy('n.lang')->addOrderBy('n.position', 'ASC');
		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('navigationsGrid:gridSortableRow!');
		$grid->setSortableGroup('group');

		// Columns
		$grid->addColumnLink('title', 'default.title', 'Default:edit');
		$grid->addColumnText('alias', 'default.alias');
		$grid->addColumnText('group', 'default.group', 'group.title');
		$grid->addColumnText('lang', 'default.lang');
		$grid->addColumnStatus('isHomepage', 'navigations.navigation.isHomepage')->setAlign('center')
			->addOption(1, 'navigations.navigation.setAsHomepage')->setIcon('star')->setShowTitle(false)->endOption()
			->addOption(0, '')->setIcon('star far')->setClass('btn-default')->setClassInDropdown('hidden')->endOption()
			->onChange[] = [$this, 'gridHomepageChange'];
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'navigations.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'navigations.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filter
		$groups = ['' => ''];
		foreach ($this->navigationsGroupsService->getAll() as $group)
			$groups[$group->getId()] = $group->title;

		$grid->addFilterText('title', 'default.title');
		$grid->addFilterText('alias', 'default.alias');
		$grid->addFilterSelect('group', 'default.group', $groups, 'g.id');
		$grid->addFilterSelect('lang', 'default.lang', [], 'n.id');

		// Actions
		$grid->addAction('edit', '', 'Default:edit')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax');

		// Columns prototype
		$grid->getColumn('isPublished')->getElementPrototype('th')->class[] = 'w1';
		$grid->getColumn('isHomepage')->getElementPrototype('th')->class[]  = 'w1';


		return $grid;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->navigationsService->removeNavigation($id))
			$presenter->flashMessageSuccess('default.removed');
		else
			$presenter->flashMessageDanger('default.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->navigationsService->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('default.publishChanged');
		else
			$presenter->flashMessageDanger('default.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function gridHomepageChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($newStatus == 1 && $this->navigationsService->setHomepage($id))
			$presenter->flashMessageSuccess('navigations.navigationForm.homepageChanged');
		else
			$presenter->flashMessageDanger('navigations.navigationForm.homepageChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleGridSortableRow()
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id', null);
		$position  = $request->getPost('position', null);

		if ($id != null && $position != null && $this->navigationsService->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$presenter->redrawControl('flashes');
	}
}
