<?php declare(strict_types = 1);

namespace Navigations\Model;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TTranslator;
use Core\FrontModule\Model\Redirects;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Entities\NavigationGroup;
use Navigations\Model\Providers\INavigationItem;
use Nette\Caching\Cache;

class Navigations extends BaseEntityService
{
	use TTranslator;

	const CACHE_NAMESPACE = 'navigations';

	protected $entityClass = Navigation::class;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => [self::CACHE_NAMESPACE],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	/** @var ItemsCollector @inject */
	public $itemsCollector;

	/** @var Redirects @inject */
	public $redirectsService;

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	/**
	 * @return Navigation|null|object
	 */
	public function getHomepageEntity()
	{
		$key = self::CACHE_NAMESPACE . '/homepageEntity.' . $this->translator->getLocale();

		return $this->getCache()->load($key, function(&$dep) {
			$dep = $this->cacheDep;

			return $this->getEr()->findOneBy(['isHomepage' => 1, 'lang' => [$this->translator->getLocale(), null]]);
		});
	}

	/**
	 * @return null|Navigation[]
	 */
	public function getAll()
	{
		$key = self::CACHE_NAMESPACE . '/all';

		return $this->getCache()->load($key, function(&$dep) {
			$dep = $this->cacheDep;

			return $this->getEr()->findAll();
		});
	}

	/**
	 * @param int $id
	 *
	 * @return null|Navigation[]
	 */
	public function getByGroup($id)
	{
		$key = self::CACHE_NAMESPACE . '/group/' . $id;

		return $this->getCache()->load($key, function(&$dep) use ($id) {
			$dep = $this->cacheDep;

			return $this->getEr()->findBy(['group' => $id]);
		});
	}

	/**
	 * @return Navigation[]|null
	 */
	public function getPublished()
	{
		$key = self::CACHE_NAMESPACE . '/published/l-' . $this->translator->getLocale();

		return $this->getCache()->load($key, function(&$dep) {
			$dep = $this->cacheDep;

			$navigations = $this->getEr()->createQueryBuilder('n')->where('n.isPublished = 1')
				->andWhere('n.lang IS NULL OR n.lang = :locale')->setParameter('locale', $this->translator->getLocale())
				->leftJoin('n.parent', 'p')->addSelect('p')
				->orderBy('n.position')->getQuery()->getResult();

			if ($navigations) {
				$gIds = [];
				foreach ($navigations as $nav) {
					$gIds[$nav->group->getId()] = $nav->group->getId();
				}
				$this->em->getRepository(NavigationGroup::class)->createQueryBuilder('g')
					->andWhere(is_array($gIds) ? 'g.id IN (:gid)' : 'g.id = :gid')->setParameter('gid', $gIds)
					->getQuery()->getResult();
			}

			return $navigations;
		});
	}

	/**
	 * Vytvoří strukturu navigace složenou z DaoNavigationItem pro komponentu navigace
	 * TODO dalo by se určitě lépe optimalizovat
	 *
	 * @return array
	 */
	public function getPublishedToDaoNavigationItem()
	{
		$key = self::CACHE_NAMESPACE . '/publishedDaoNavigationItem/l-' . $this->translator->getLocale();

		return $this->getCache()->load($key, function(&$dep) {
			$dep = $this->cacheDep;

			$flat = [];
			$data = [];

			// Načtení dat z DB a složení do pole pro pozdější práci
			foreach ($this->getPublished() as $nav) {
				$flat[] = [
					'id'       => $nav->getId(),
					'parent'   => $nav->getParent() ? $nav->getParent()->getId() : 0,
					'nav'      => $nav,
					'children' => [],
				];
			}

			// Funkce pro fložení dat do Dao
			$fn = function($item, $dao = null) use (&$fn) {
				$nav = $item['nav'];
				if (!$dao) {
					$dao = new DaoNavigationItem($nav);
				}

				$dao->set('id', $nav->getId());
				$dao->set('link', $this->generateLink($nav));
				$dao->set('class', trim('item-' . $nav->getId() . ($nav->getParam('linkClass') ?: '')));

				foreach ($item['children'] as $child) {
					$daoC = new DaoNavigationItem($child['nav']);
					$daoC = $fn($child, $daoC);

					if ($daoC) {
						$dao->addChild($daoC);
					}
				}

				return $dao;
			};

			// Vytvoření stromové struktury navigace
			foreach (Arrays::buildTree($flat, 'parent', 'id') as $item) {
				$dao = $fn($item);

				if ($dao) {
					$data[$item['nav']->group->type][] = $dao;
				}
			}

			return $data;
		});
	}

	/**
	 * @param $id
	 *
	 * @return array|mixed
	 */
	public function getPublishedByGroup($id)
	{
		$key = self::CACHE_NAMESPACE . '/publishedByGroup/' . $id . '/l-' . $this->translator->getLocale();

		return $this->getCache()->load($key, function(&$dep) use ($id) {
			$dep = $this->cacheDep;

			return $this->em->createQueryBuilder()->select('n')->from(Navigation::class, 'n')
				->where('n.group = :group')->andWhere('n.isPublished = 1')->setParameter('group', $id)
				->andWhere('n.lang IS null OR n.lang = :locale')->setParameter('locale', $this->translator->getLocale())
				->orderBy('n.position')->getQuery()->getResult();
		});
	}

	/**
	 * @param string $component
	 *
	 * @return mixed
	 */
	public function getPublishedByComponent($component)
	{
		$key = self::CACHE_NAMESPACE . '/publishedByComponent/' . $component . '/l-' . $this->translator->getLocale();

		return $this->getCache()->load($key, function(&$dep) use ($component) {
			$dep = $this->cacheDep;

			return $this->getEr()->createQueryBuilder('n')->where('n.isPublished = 1')
				->andWhere('n.componentType = :component')->setParameter('component', $component)
				->andWhere('n.lang IS null OR n.lang = :locale')->setParameter('locale', $this->translator->getLocale())
				->getQuery()->getResult();
		});
	}

	/**
	 * @param $id
	 *
	 * @return null|object|Navigation
	 */
	public function getNavigation($id)
	{
		$key = self::CACHE_NAMESPACE . '/nav/' . $id;

		return $this->getCache()->load($key, function(&$dep) use ($id) {
			$dep = $this->cacheDep;

			return $this->getEr()->find($id);
		});
	}

	/**
	 * @param $alias
	 *
	 * @return Navigation|null|object
	 */
	public function findByAlias($alias)
	{
		$locale = is_array($alias) && isset($alias['locale']) ? $alias['locale'] : null;
		if (!is_string($alias))
			$alias = is_array($alias) && !empty($alias) && isset($alias['path']) ? $alias['path'] : null;

		if (!$alias)
			return null;

		$navigations = $this->getEr()->createQueryBuilder('n')
			->andWhere('n.alias = :alias')->setParameter('alias', $alias)
			->andWhere('n.lang IS NULL OR n.lang = :locale')->setParameter('locale', $locale)
			->andWhere('n.isPublished = 1')
			->getQuery()->getResult();

		$navigation = null;

		foreach ($navigations as $v)
			if ($v->lang == $locale)
				$navigation = $v;

		if (!$navigation)
			$navigation = array_values($navigations)[0];

		// Vyhledání v přesměrování
		if (!$navigation) {
			$redirect = $this->redirectsService->getEr()->findOneBy(['from'        => $alias,
			                                                         'package'     => 'Navigations',
			                                                         'relationKey' => 'Navigation']);

			if ($redirect)
				$navigation = $this->get($redirect->relationValue);
		}

		return $navigation;
	}

	/**
	 * Zpracování navigace a získání URL
	 *
	 * @param Navigation $navigation
	 *
	 * @return array|bool
	 */
	public function generateLink(Navigation $navigation)
	{
		$t = ['activeNavigation' => $navigation];

		return $this->getUrlById($t);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function getHomepage($locale = null)
	{
		try {
			$locale = $locale ?? $this->translator->getLocale();
			$key    = self::CACHE_NAMESPACE . '/route/hp/' . $locale;

			return $this->getCache()->load($key, function(&$dep) use ($key, $locale) {
				$dep = $this->cacheDep;

				$navigation = $this->getEr()->findOneBy(['isHomepage' => 1, 'isPublished' => 1,
				                                         'lang'       => [null, $locale]]);

				if (!$navigation || !($component = $this->itemsCollector->getItemById($navigation->componentType)))
					return null;

				return $component->routerIn($navigation->componentParams) + ['activeNavigation' => $navigation];
			});
		} catch (\Exception $e) {
		}

		return null;
	}

	public function getIdByUrl($url)
	{
		try {
			$return = [];
			$locale = $url['locale'] ?? $this->translator->getLocale();
			$key    = self::CACHE_NAMESPACE . '/idByUrl/' . serialize($url) . '/l-' . $locale;

			return $this->getCache()->load($key, function(&$dep) use ($url, $return, $locale) {
				$dep = $this->cacheDep;

				$navigation = null;

				if (isset($url['path'])) {
					$tmp = explode('/', $url['path']);
					$t   = $tmp;

					// Upravení pro stránkování u obecné komponenty
					$paginatorIndex = array_search($this->t('default.urlPart.page'), $t);
					if ($paginatorIndex !== false) {
						$return['list-page'] = $t[$paginatorIndex + 1] ?? 1;
						$return['do']        = 'list-paginator';
						unset($t[$paginatorIndex + 1]);
						unset($t[$paginatorIndex]);
						unset($tmp[$paginatorIndex + 1]);
						unset($tmp[$paginatorIndex]);
					}

					while ($t && !$navigation) {
						$navigation = $this->findByAlias(['path' => array_pop($t), 'locale' => $locale]);
					}

					$url['path'] = implode('/', $tmp);
				}

				if ($navigation && $component = $this->itemsCollector->getItemById($navigation->componentType)) {
					$return += $component->routerIn($navigation, $url) + ['activeNavigation' => $navigation];
				} else if (!$navigation) {
					$tmp       = $this->getHomepage();
					$component = $this->itemsCollector->getItemById($tmp['activeNavigation']->componentType);

					if (!$component)
						return null;
					$return += $component->routerIn($tmp['activeNavigation']->componentParams, $url);

					if ($return['id'] == null)
						return null;

					$tmp['activeNavigation']->isHomepage = 0;

					$return += ['activeNavigation' => $tmp['activeNavigation']];
				}

				return $return;
			});
		} catch (\Exception $e) {
			return null;
		}
	}

	public function getUrlById(&$params)
	{
		$return = null;
		try {
			$key = self::CACHE_NAMESPACE . '/idByUrl/' . serialize($params);

			return $this->getCache()->load($key, function(&$dep) use ($params, $return) {
				$dep = $this->cacheDep;

				if (isset($params['activeNavigation'])) {
					$navType = $params['activeNavigation']->componentType;
					if ($navType == 'navigation.alias') {
						$tmp = $this->getNavigation($params['activeNavigation']->componentParams['navigation']);

						if ($tmp) {
							$params['activeNavigation'] = $tmp;
							$navType                    = $tmp->componentType;
						}
					}
					$component = $this->itemsCollector->getItemById($navType);

					$return = $this->getPreAlias($params['activeNavigation']) . $component->routerOut($params['activeNavigation'], $params);
				}

				if (!$return) {
					$groups = $this->itemsCollector->findItemByPresenter($params['presenter'], $params['action']);

					if (!$groups) {
						if ($params['presenter']) {
							preg_match('/Override$/', $params['presenter'], $matches, PREG_OFFSET_CAPTURE, 0);

							if (isset($matches[0]))
								$params['presenter'] = substr($params['presenter'], 0, $matches[0][1]);
						}

						$groups = $this->itemsCollector->findItemByPresenter($params['presenter'], $params['action']);
					}

					if (!$groups)
						return null;

					ksort($groups);

					foreach ($groups as $components) {
						/** @var INavigationItem $components */
						foreach ($components as $component) {
							foreach ($this->getPublishedByComponent($component->getId()) as $navigation) {
								$return = $component->routerOut($navigation, $params);

								if ($return === '')
									$return = null;
								else if ($return !== null && $return !== false)
									$return = $this->getPreAlias($navigation) . $return;
							}
						}
					}
				}

				// Upravení url pro stránkování
				if (isset($params['do']) && isset($params['list-page']) && $params['do'] == 'list-paginator') {
					if ($params['list-page'] > 1) {
						$return .= '/' . $this->t('default.urlPart.page') . '/' . $params['list-page'];
					}

					unset($params['list-page']);
					unset($params['do']);
				}

				if (isset($params['activeNavigation']))
					$params['activeNavigation']->link = $return;

				return $return;
			});
		} catch (\Exception $e) {
		}

		return null;
	}

	/**
	 * @param Navigation $navigation
	 *
	 * @return string
	 */
	public function getPreAlias($navigation)
	{
		$arr = [];
		$t   = $navigation->parent;

		while ($t) {
			$arr[] = $t->alias;
			$t     = $t->parent;
		}

		if ($navigation->getLang() && $this->translator->getDefaultLocale() != $navigation->getLang())
			$arr[] = $navigation->getLang();

		return $arr ? '/' . implode('/', array_reverse($arr)) : '';
	}

	/*******************************************************************************************************************
	 * ===========================  Cache
	 */
}
