<?php declare(strict_types = 1);

namespace Navigations\AdminModule\Model;

use Core\Model\Entities\Repository\NestedTreeRepository;
use Core\Model\Exceptions\EntityContainChildren;
use Core\Model\Helpers\BaseEntityService;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Entities\NavigationGroup;
use Navigations\Model\Entities\NavigationText;
use Nette\Caching\Cache;

/**
 * Class Navigations
 * @package Navigations\AdminModule\Model
 *
 * @method Navigation|object|null getReference($id)
 * @method Navigation[]|null getAll()
 * @method Navigation|null get($id)
 * @method NestedTreeRepository getEr()
 */
class Navigations extends BaseEntityService
{
	/** @var string */
	protected $entityClass = Navigation::class;

	/** @var string */
	protected $entityClassText = NavigationText::class;

	const CACHE_NAMESPACE = 'navigations';

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	public function cleanCache()
	{
		$this->getCache()->clean([Cache::TAGS => [self::CACHE_NAMESPACE]]);
	}

	public function setHomepage(int $id, string $lang): bool
	{
		$nav = $this->get($id);

		if (!$nav || !$nav->getText($lang))
			return false;

		$this->em->beginTransaction();
		try {
			$this->em->createQueryBuilder()->update($this->entityClassText, 'nt')
				->set('nt.isHomepage', 0)
				->where('nt.isHomepage = 1')->andWhere('nt.lang = :lang')
				->setParameter('lang', $lang)
				->getQuery()->execute();

			$nav->getText($lang)->isHomepage = 1;
			$this->em->persist($nav->getText($lang))->flush();
			$this->em->commit();
		} catch (\Exception $e) {
			$this->em->rollback();

			return false;
		}

		return true;
	}

	public function invertPublish(int $id, string $lang)
	{
		$nav = $this->get($id);

		if (!$nav || !$nav->getText($lang))
			return false;

		$nav->getText($lang)->isPublished = !$nav->getText($lang)->isPublished;
		$this->em->persist($nav->getText($lang))->flush();

		return true;
	}

	public function removeNavigation(int $id): bool
	{
		if ($nav = $this->get($id)) {
			if ($nav->getChildren()->count())
				throw new EntityContainChildren('navigations.navigation.removeFailedHasChilds');
			$this->em->remove($nav);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function findByAlias(string $alias, ?string $lang = null, ?int $compareParent = null)
	{
		$qb = $this->getEr()->createQueryBuilder('n')
			->andWhere('nt.alias = :alias')
			->setParameter('alias', $alias);

		if ($lang) {
			$qb->join('n.texts', 'nt', 'WITH', 'nt.lang = :lang')
				->setParameter('lang', $lang);
		} else {
			$qb->join('n.texts', 'nt');
		}

		if ($compareParent) {
			$qb->andWhere('n.parent = :parent')
				->setParameter('parent', $compareParent);
		}

		return $qb->getQuery()->getResult();
	}

	/**
	 * @return array
	 * @throws \Exception
	 */
	public function fixTreeStructure()
	{
		$result = ['status' => 'error'];

		$groups  = $this->em->getRepository(NavigationGroup::class)->createQueryBuilder('ng')->select('ng.title, ng.id')->getQuery()->getArrayResult();
		$parents = [];
		/** @var Navigation[] $navs */
		$navs = $this->getEr()->createQueryBuilder('n')->orderBy('n.parent')->addOrderBy('n.position', 'DESC')->getQuery()->getResult();
		/** @var Navigation[] $roots */
		$roots = [];

		// Vytvoření dočasného rootu
		$tmp     = $this->em->getRepository(NavigationGroup::class)->findOneBy([]);
		$tmpRoot = new Navigation('ROOT - TMP', $this->em->getReference(NavigationGroup::class, $tmp->getId()), '', 'navigation.customLink', 0);
		$this->em->persist($tmpRoot);
		$this->em->flush();

		// Získání všech rootů
		foreach ($groups as $g) {
			$root = $this->getEr()->findOneBy(['title' => 'ROOT - ' . $g['title']]);
			if (!$root) {
				$root = new Navigation('ROOT - ' . $g['title'], $this->em->getReference(NavigationGroup::class, $g['id']), '', 'navigation.customLink', 0);
				$this->em->persist($root);
				$this->em->flush();
			}
			$this->getEr()->createQueryBuilder('n')->update()
				->set('n.root', ':root')
				->where('n.id = :id')
				->getQuery()->execute(['root' => $root->getId(), 'id' => $root->getId()]);
			$roots[$g['id']] = $root;
		}

		// Získání všech navigací bez rootu
		foreach ($navs as $k => $n) {
			if ($roots[$n->group->getId()]->getId() == $n->getId() || strpos($n->title, 'ROOT - ') === 0) {
				unset($navs[$k]);
				continue;
			}

			if ($n->getParent())
				$parents[$n->getId()] = $n->getParent();
		}

		// Vložení do dočasného rootu
		foreach ($navs as $n) {
			$this->getEr()->persistAsFirstChildOf($n, $tmpRoot);
			$this->em->flush();
		}

		$this->getEr()->verify();

		// Vložení do správných rootů
		foreach ($navs as $n) {
			$this->getEr()->persistAsFirstChildOf($n, $roots[$n->group->getId()]);
			$this->em->flush();
		}

		// Nastavení nadřazených
		foreach ($navs as $n) {
			if (isset($parents[$n->getId()])) {
				$this->getEr()->persistAsFirstChildOf($n, $parents[$n->getId()]);
				$this->em->flush();
			}
		}

		$errors = $this->getEr()->verify();
		$this->em->flush();

		$this->em->remove($tmpRoot);
		$tmpRoots = $this->getEr()->findBy(['title' => 'ROOT - TMP']);
		if ($tmpRoots)
			$this->em->remove($tmpRoots);
		$this->em->flush();

		if (is_array($errors))
			$result['errors'] = $errors;
		else
			$result['status'] = 'ok';

		return $result;
	}
}
