<?php declare(strict_types = 1);

namespace Navigations\AdminModule\Components;

use Core\FrontModule\Model\SeoContainer;
use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\FormValidateEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\CheckboxListInput;
use Core\Model\UI\Form\Controls\SelectInput;
use Navigations\Model\BaseItem;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Entities\NavigationText;
use Navigations\Model\ItemsCollector;
use Navigations\AdminModule\Model\Navigations;
use Navigations\AdminModule\Model\NavigationsGroups;
use Nette\Http\IResponse;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;
use Nette\Utils\Strings;

/**
 * TODO předělat načítání componentType, aby bylo bez ajaxu jako je u TemplatePages
 *
 * Class NavigationForm
 * @package Navigations\AdminModule\Components
 */
class NavigationForm extends BaseControl
{
	/** @var int @persistent */
	public $navId;

	/** @var Navigation */
	public $navigation;

	/** @var ItemsCollector */
	protected $itemsCollector;

	/** @var NavigationsGroups */
	protected $navigationsGroups;

	/** @var Navigations */
	protected $navigationsService;

	/** @var SeoContainer */
	protected $seoContainerService;

	public function __construct(ItemsCollector $itemsCollector, NavigationsGroups $navigationsGroups, Navigations $navigations,
	                            SeoContainer $seoContainer)
	{
		$this->itemsCollector      = $itemsCollector;
		$this->navigationsGroups   = $navigationsGroups;
		$this->navigationsService  = $navigations;
		$this->seoContainerService = $seoContainer;
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];
		$this->eventDispatcher->dispatch(NavigationGroupForm::class . '::render', new ComponentTemplateEvent($this->template, $this));
		$this->template->render($this->getTemplateFile() ?: __DIR__ . '/NavigationForm.latte');
	}

	protected function attached($presenter): void
	{
		parent::attached($presenter);

		/** @var Request $httpRequest */
		$httpRequest = $presenter->getHttpRequest();

		$componentType = $httpRequest->getPost('componentType');
		if ($componentType)
			$this->loadComponent($componentType);

		$navGroupPost = $httpRequest->getPost('navigationGroup') ?: $this['form']['navigationGroup']->getValue();
		$this->loadParentItems($navGroupPost);
	}

	protected function loadParentItems($groupId): void
	{
		$root = $this->navigationsService->getEr()->findOneBy(['group' => $groupId, 'parent' => null, 'lvl' => 0]);
		if ($root) {
			$parents = [$root->getId() => ''];
			$q       = $this->navigationsService->getEr()->createQueryBuilder('n')
				->select('n.id, nt.title, n.lvl')
				->join('n.texts', 'nt', 'WITH', 'nt.lang = :lang')
				->where('n.lvl > 0')->andWhere('n.group = :gId')
				->orderBy('n.root')->addOrderBy('n.lft')
				->setParameters([
					'gId'  => $groupId,
					'lang' => $this->translator->getLocale(),
				]);

			if ($this->navigation) {
				$q->andWhere('n.id != :curId')->setParameter('curId', $this->navigation->getId());
			}
			foreach ($q->getQuery()->getScalarResult() as $v) {
				$title = ' ' . $v['title'];
				for ($i = 1; $i < $v['lvl']; $i++)
					$title = '---' . $title;
				$parents[$v['id']] = $title;
			}
			$this['form']['parent']->setItems($parents);
		}
	}

	protected function loadComponent($componentId): void
	{
		$form = $this['form'];

		if ($form->getComponent('component', false)) {
			$form->removeComponent($form['component']);
		}

		$component = $this->itemsCollector->getItemById($componentId);

		if ($component) {
			$form->addComponent($component->getFormContainer(), 'component');
			$component = $form['component'];
			if ($this->navigation) {
				foreach ($component->components as $k => $c) {
					$value = $this->navigation->componentParams[$k] ?? null;

					if (!$value)
						continue;

					if ($c instanceof SelectInput) {
						if (array_key_exists($value, $c->getItems())) {
							$c->setDefaultValue($value);
						}
					} else if ($c instanceof CheckboxListInput) {
						$tmp = [];
						foreach ($value as $v) {
							if (array_key_exists($v, $c->getItems()))
								$tmp[] = $v;
						}

						$c->setDefaultValue($tmp);
					} else {
						$c->setDefaultValue($value);
					}
				}
			}
		}
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleLoadInputs(string $componentId): void
	{
		$presenter = $this->getPresenter();

		try {
			$this->loadComponent($componentId);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (\Exception $e) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('component');
		$presenter->redrawControl('flashes');
	}

	public function handleLoadParents($group): void
	{
		$presenter = $this->getPresenter();

		try {
			$this->loadParentItems($group);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (\Exception $e) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('parent');
		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$lang = $this->translator->getLocale();

		$components = ['' => $this->t('default.choose')];
		foreach ($this->itemsCollector->getItems() as $groupName => $group) {
			$arr = [];

			foreach ($group['items'] as $key => $class) {
				/** @var $class BaseItem */
				$arr[$class->getId()] = $this->t($class->getTitle());
			}

			$components[$this->t($group['title'])] = $arr;
		}

		$navGroups = [];
		foreach ($this->navigationsGroups->getAll() as $group)
			$navGroups[$group->getId()] = $group->getText()->title;

		$openIn = [
			'default' => 'navigations.openIn.default',
			'_blank'  => 'navigations.openIn.blank',
		];

		$form->addText('title', 'navigations.navigation.title')->setMaxLength(255)
			->setIsMultilanguage();
		$form->addText('alias', 'navigations.navigation.alias')->setMaxLength(255)
			->setIsMultilanguage();
		$form->addBool('isPublished', 'default.isPublished')
			->setIsMultilanguage()->setDefaultValue(0);
		$form->addSelect('openIn', 'navigations.navigation.openIn', $openIn)->setDefaultValue('default');
		$form->addSelect('navigationGroup', 'navigations.navigation.navigationGroup', $navGroups)->setDefaultValue(array_keys($navGroups)[0])->setRequired();
		$form->addSelect('parent', $this->t('default.parent'), [])->setTranslator(null);
		$form->addText('linkClass', 'navigations.navigation.linkClass');
		$form->addText('pageClass', 'navigations.navigation.pageClass');
		$form->addSelect('componentType', $this->t('navigations.navigation.component'), $components)
			->setTranslator(null)->setRequired();

		$form->addComponent($this->seoContainerService->getContainer(true), 'seo');

		$this->eventDispatcher->dispatch(NavigationGroupForm::class . '::createForm',
			new CreateFormEvent($form, $this->getPresenter(false) ? $this->template : null));

		$form->addSaveCancelControl('saveControl');

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values)
	{
		$presenter = $this->getPresenter(false);
		$nav       = $this->navId ? $this->navigationsService->get($this->navId) : null;

		$titleEmpty = true;
		foreach ($values->title as $l => $v) {
			if ($v !== '') {
				$titleEmpty = false;
				break;
			}
		}

		foreach ($values->alias as $l => $v) {
			if (!$v && $values->title[$l])
				$values->alias[$l] = Strings::webalize($values->title[$l]);
			$texts = $nav ? $nav->getTexts() : null;

			if (!$nav || (isset($texts[$l]) && $texts[$l]->getAlias() != $v)) {
				$tmpNav = $this->navigationsService->findByAlias($v, $l, $values->parent);
				if ($tmpNav)
					$form->addError($this->t('navigations.navigationForm.aliasExists') . ' (' . $l . ')');
			}
		}

		if ($titleEmpty)
			$form->addError($this->t('default.formMessages.requiredField', ['input' => $this->t($form['title']->caption)]), false);

		if ($form->hasErrors()) {
			if ($form->isAjax()) {
				$this->redrawControl('form');
			} else {
				$this->redirect('this');
			}
		}

		$this->eventDispatcher->dispatch(NavigationGroupForm::class . '::validateForm',
			new FormValidateEvent($form, $values, $presenter ? $this->template : null, $presenter ?: null));
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$presenter = $this->getPresenter();
		$this->em->beginTransaction();

		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var NavigationText[] $texts */
			$texts    = [];
			$navGroup = $this->navigationsGroups->getReference($values->navigationGroup);

			if ($this->navId) {
				$navigation = $this->navigationsService->get($this->navId);
				$texts      = $navigation->getTexts()->toArray();
			} else {
				$navigation = new Navigation($navGroup, $values->componentType);
			}

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l]) && $v['title'])
					$texts[$l] = new NavigationText($navigation, $v['title'], $l);

				if ($v['title'] == '' || $texts[$l]->getTitle() == '') {
					if ($texts[$l])
						$this->em->remove($texts[$l]);
					unset($texts[$l]);
					continue;
				}

				$texts[$l]->setTitle($v['title'])
					->setAlias($v['alias'])
					->setSeo($v['seo'] ?? [])
					->setLang($l);
				$texts[$l]->isPublished = $v['isPublished'];

				$this->em->persist($texts[$l]);
			}

			$navigation->setTexts($texts);
			$navigation->link            = '';
			$navigation->componentType   = $values->componentType;
			$navigation->componentParams = $this->getPresenter()->getRequest()->getPost('component');
			$navigation->locale          = $this->translator->getLocale();

			$navigation->setParam('linkClass', $values->linkClass);
			$navigation->setParam('pageClass', $values->pageClass);
			$navigation->setParam('openIn', $values->openIn == 'default' ? null : $values->openIn);
			$navigation->setParent($values->parent ? $this->navigationsService->getReference($values->parent) : null);
			$navigation->setGroup($values->navigationGroup ? $this->navigationsGroups->getReference($values->navigationGroup) : null);

			$event                   = new FormSuccessEvent($form, $values, $this->template, $presenter);
			$event->custom['entity'] = $navigation;
			$this->eventDispatcher->dispatch(NavigationGroupForm::class . '::formSuccess', $event);

			$flashMessage = $navigation->getId() ? 'navigations.navigationForm.edited' : 'navigations.navigationForm.added';
			$this->em->persist($navigation);
			$this->em->flush();
			$form->addCustomData('navigationId', $navigation->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
			$this->em->commit();
		} catch (\Exception $e) {
			$this->em->rollback();
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ===========================  Get/Set
	 */

	public function setNavigation(int $id): void
	{
		$this->navId      = $id;
		$this->navigation = $this->navigationsService->get($id);

		if (!$this->navigation)
			$this->getPresenter()->error();

		$form = $this['form'];
		$n    = $this->navigation;
		$this->loadParentItems($n->getGroup()->getId());

		$d = [];

		$openIn = $n->getParam('openIn');
		if ($openIn && array_key_exists($openIn, $form['openIn']->getItems()))
			$d['openIn'] = $openIn;
		if ($n->getGroup() && array_key_exists($n->getGroup()->getId(), $form['navigationGroup']->getItems()))
			$d['navigationGroup'] = $n->getGroup()->getId();
		if ($n->getParent() && array_key_exists($n->getParent()->getId(), $form['parent']->getItems()))
			$d['parent'] = $n->getParent()->getId();
		if ($n->componentType && array_key_exists($n->componentType, $form['componentType']->getItems())) {
			$d['componentType']                                                  = $n->componentType;
			$form['componentType']->getControlPrototype()->attrs['data-default'] = Json::encode($n->componentParams);
		}

		foreach ($n->getParams() as $k => $v) {
			if (isset($form[$k]))
				$d[$k] = $v;
		}

		if ($n->componentType)
			$this->loadComponent($n->componentType);

		foreach ($n->getTexts() as $l => $text) {
			$d['title'][$l]       = $text->getTitle();
			$d['alias'][$l]       = $text->getAlias();
			$d['isPublished'][$l] = $text->isPublished;
		}
		$this->seoContainerService->setDefaultsFromEntity($form['seo'], $n->getTexts()->toArray());

		$form->setDefaults($d);
	}
}
