<?php declare(strict_types = 1);

namespace Navigations\Model\Entities;

use Core\Model\Entities\TSeo;
use Doctrine\ORM\Mapping as ORM;
use Nette\Utils\Strings;

/**
 * Class NavigationText
 * @package Navigations\Model\Entities
 * @ORM\Table(name="navigations__navigation_text", indexes={@ORM\Index(name="navigation", columns={"lang", "navigation_id", "is_published"})})
 * @ORM\Entity
 * @ORM\EntityListeners({"NavigationTextListener"})
 */
class NavigationText
{
	use TSeo;

	/**
	 * @var Navigation
	 * @ORM\Id
	 * @ORM\ManyToOne(targetEntity="Navigation", inversedBy="texts")
	 * @ORM\JoinColumn(name="navigation_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public $navigation;

	/**
	 * @var string
	 * @ORM\Id
	 * @ORM\Column(name="lang", type="string", length=2, nullable=true)
	 */
	protected $lang;

	/**
	 * @var string
	 * @ORM\Column(name="title", type="string", length=255, nullable=false)
	 */
	public $title;

	/**
	 * @var string
	 * @ORM\Column(name="alias", type="string", length=255, nullable=false)
	 */
	protected $alias;

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint", nullable=false, options={"default":1})
	 */
	public $isPublished;

	/**
	 * @var int
	 * @ORM\Column(name="is_homepage", type="smallint", nullable=false, options={"default":0})
	 */
	public $isHomepage;

	/**
	 * @var string
	 * @ORM\Column(name="img", type="string", length=400, nullable=true)
	 */
	public $img;

	/**
	 * @var string
	 * @ORM\Column(name="description", type="text", nullable=true)
	 */
	public $description;

	/**
	 * @var array
	 * @ORM\Column(name="params", type="array", nullable=true)
	 */
	public $params;

	/**
	 * @var string
	 * @ORM\Column(name="custom_text1", type="string", length=255, nullable=true)
	 */
	public ?string $customText1 = null;

	/**
	 * @var string
	 * @ORM\Column(name="custom_text2", type="string", length=255, nullable=true)
	 */
	public ?string $customText2 = null;

	/**
	 * @ORM\Column(name="full_url", type="string", nullable=true)
	 */
	public ?string $fullUrl = null;

	public function __construct(Navigation $navigation, string $title, ?string $img = null, ?string $description, string $lang)
	{
		$this->navigation  = $navigation;
		$this->lang        = $lang;
		$this->title       = $title;
		$this->isPublished = 1;
		$this->isHomepage  = 0;
		$this->img         = $img;
		$this->description = $description;
		$this->params      = [];
		$this->setSeo([]);
	}

	public function getLang(): string
	{
		return $this->lang;
	}

	public function setLang(?string $lang): self
	{
		$this->lang = $lang === '' ? null : $lang;

		return $this;
	}

	public function getTitle(): string
	{
		return $this->title;
	}

	public function setTitle(string $title): self
	{
		$this->title = $title;

		if (!$this->alias)
			$this->setAlias($title);

		return $this;
	}

	public function getAlias(): ?string
	{
		return $this->alias;
	}

	public function setAlias(string $alias): self
	{
		if ($alias !== '')
			$this->alias = Strings::webalize($alias);
		else
			$this->alias = Strings::webalize($this->title);

		return $this;
	}

	/**
	 * @return string
	 */
	public function getImg()
	{
		return $this->img;
	}

	/**
	 * @param string $img
	 */
	public function setImg($img)
	{
		$this->img = $img;

		return $this;
	}

	/**
	 * @return string
	 */
	public function getDescription()
	{
		return $this->description;
	}

	/**
	 * @param string $description
	 */
	public function setDescription($description)
	{
		$this->description = $description;

		return $this;
	}

	public function getParams(): array
	{
		return $this->params ?: [];
	}

	public function getParam($key)
	{
		return $this->params[$key] ?? null;
	}

	public function setParam(string $key, $value): self
	{
		if ($value) {
			$this->params[$key] = $value;
		} else {
			unset($this->params[$key]);
		}

		return $this;
	}
}
