<?php declare(strict_types = 1);

namespace Navigations\Model;

use Core\Model\Event\EventDispatcher;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Helpers\BaseEntityService;
use Core\FrontModule\Model\Redirects;
use Core\Model\Navigation\Alias;
use Core\Model\Navigation\CustomLink;
use Core\Model\Parameters;
use Navigations\Model\Entities\NavigationGroup;
use Navigations\Model\Entities\NavigationGroupText;
use Navigations\Model\Entities\NavigationText;
use Navigations\Model\Event\RouteInEvent;
use Gedmo\Tree\Entity\Repository\NestedTreeRepository;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Helper\NavigationsHelper;
use Navigations\Model\Providers\INavigationItem;
use Nette\Caching\Cache;
use Nette\Localization\ITranslator;
use Nette\Utils\Strings;

/**
 * Class Navigations
 * @package Navigations\Model
 *
 * @method Navigation|object|null getReference($id)
 * @method Navigation|null get($id)
 * @method NestedTreeRepository getEr()
 */
class Navigations extends BaseEntityService
{
	/** @var string */
	const CACHE_NAMESPACE = 'navigations';

	/** @var string */
	protected $entityClass = Navigation::class;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => [self::CACHE_NAMESPACE],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => false,
	];

	/** @var DaoNavigationItem[][] */
	public $cNavs;

	/** @var DaoNavigationItem[][] */
	public $cNavsTree;

	protected array $cUrlById = [];
	protected array $cIdByUrl = [];

	/** @var ITranslator @inject */
	public $translator;

	/** @var ItemsCollector @inject */
	public $itemsCollector;

	/** @var Redirects @inject */
	public $redirectsService;

	/** @var EventDispatcher @inject */
	public $eventDispatcher;

	/** @var NavigationsHelper @inject */
	public $navigationsHelper;

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	protected function getCurrentSiteIdent(): string { return $this->navigationsHelper->getSitesService()->getCurrentSite()->getIdent(); }

	/**
	 * @param int    $id
	 * @param string $lang
	 *
	 * @return DaoNavigationItem[]
	 */
	public function getPath(int $id, ?string $lang = null): array
	{
		$lang = $lang ?: $this->translator->getLocale();

		$result = [];
		$stmt   = $this->em->getConnection()->prepare("
						SELECT T2.id 
						FROM (
							SELECT @r AS _id,
								(SELECT @r := parent_id FROM navigations__navigation WHERE id = _id) AS parent_id, @l := @l + 1 AS lvl 
								FROM (SELECT @r := {$id}, @l := 0) vars, navigations__navigation h WHERE @r <> 0) T1 
						JOIN navigations__navigation T2 ON T1._id = T2.id ORDER BY T1.lvl DESC");
		$stmt->execute();

		foreach ($stmt->fetchAll() as $row) {
			$nav = $this->getNavigation((int) $row['id'], $lang);
			if ($nav)
				$result[$row['id']] = $nav;
		}

		return $result;
	}

	/**
	 * @param string $lang
	 *
	 * @return DaoNavigationItem[]
	 */
	public function getPublishedToDaoNavigationItem(?string $lang = null): array
	{
		$site = $this->getCurrentSiteIdent();
		$lang = $lang ?: $this->translator->getLocale();

		if (!isset($this->cNavs[$site][$lang])) {
			$key = 'publishedNavs/' . $this->getCurrentSiteIdent() . '/' . $lang;

			$this->cNavs[$site][$lang] = $this->getCache()->load($key, function(&$dep) use ($lang) {
				$dep    = $this->cacheDep;
				$result = [];

				$loop = function(array $arr) use (&$loop, &$result) {
					/** @var DaoNavigationItem[] $arr */
					foreach ($arr as $v) {
						$result[$v->id] = $v;

						if ($v->childs)
							$loop($v->childs);
					}
				};

				foreach ($this->getPublishedDaoNavigationStructure($lang) as $v)
					$loop($v);

				return $result;
			});
		}

		return $this->cNavs[$site][$lang];
	}

	public function getPublishedDaoNavigationStructure(?string $lang = null): array
	{
		$site = $this->getCurrentSiteIdent();
		$lang = $lang ?: $this->translator->getLocale();

		if (!isset($this->cNavsTree[$site][$lang])) {
			$key = 'publishedStructure/' . $this->getCurrentSiteIdent() . '/' . $lang;

			$this->cNavsTree[$site][$lang] = $this->getCache()->load($key, function(&$dep) use ($lang) {
				$dep = $this->cacheDep;
				/** @var DaoNavigationItem[] $flat */
				$flat    = [];
				$grouped = [];
				$tree    = [];

				foreach ($this->navigationsHelper->getPublished($lang) as $nav) {
					$grouped[$nav->groupType][$nav->parentId][$nav->id] = $nav;
					$flat[$nav->id]                                     = &$grouped[$nav->groupType][$nav->parentId][$nav->id];
				}
				$this->em->clear([
					Navigation::class,
					NavigationText::class,
					NavigationGroup::class,
					NavigationGroupText::class,
				]);

				$createTree = function(&$list, $parent) use (&$createTree) {
					$tree = [];
					foreach ($parent as $k => $l) {
						if (isset($list[$l->id])) {
							if ($l->lvl >= 1)
								foreach ($list[$l->id] as $kk => $v) {
									$list[$l->id][$kk]->parent = &$parent[$k];
								}
							$l->isParent = true;
							$l->childs   = $createTree($list, $list[$l->id]);
						}

						$tree[] = $l;
					}

					return $tree;
				};

				foreach ($this->getEr()->createQueryBuilder('n')->select('n.id, ng.type')
					         ->join('n.group', 'ng')
					         ->where('n.lvl = 0')
					         ->andWhere('n.site = :site')
					         ->setParameters([
						         'site' => $this->getCurrentSiteIdent(),
					         ])->getQuery()->getScalarResult() as $row) {
					if (!isset($grouped[$row['type']]))
						continue;

					$dao                        = new DaoNavigationItem();
					$dao->id                    = $row['id'];
					$dao->parent                = 0;
					$grouped[$row['type']][0][] = $dao;

					$tree[$row['type']] = $createTree($grouped[$row['type']], $grouped[$row['type']][0])[0]->childs;
				}

				$aliases = [];

				foreach ($flat as $k => $v) {
					if ($v->componentType == Alias::TYPE) {
						$aliases[$k] = $v;
						continue;
					}

					if ($flat[$k]->link)
						$flat[$k]->link = $this->getPreAlias($flat[$k]) . $flat[$k]->link;
					else
						$flat[$k]->link = $this->generateLink($v);

					if ($flat[$k]->link && Strings::startsWith($flat[$k]->link, '/http'))
						$flat[$k]->link = substr($flat[$k]->link, 1);
				}

				foreach ($aliases as $k => $v) {
					$v->link  = $flat[$v->componentParams['navigation'][$lang]]->link;
					$flat[$k] = $v;
				}

				return $tree;
			});
		}

		return $this->cNavsTree[$site][$lang];
	}

	/**
	 * @param string $component
	 *
	 * @return DaoNavigationItem[]
	 */
	public function getPublishedByComponent(string $component, ?string $lang = null): array
	{
		$result = [];
		foreach ($this->getPublishedToDaoNavigationItem($lang) as $nav) {
			if ($nav->componentType === $component)
				$result[$nav->id] = $nav;
		}

		return $result;
	}

	public function getNavigation(int $id, ?string $lang = null): ?DaoNavigationItem
	{
		return $this->getPublishedToDaoNavigationItem($lang)[$id] ?? null;
	}

	public function generateLink(DaoNavigationItem $navigation): ?string
	{
		$t = [
			'locale'           => $navigation->getLang(),
			'activeNavigation' => $navigation,
		];

		return $this->getUrlById($t);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function getHomepage(?string $lang = null): ?DaoNavigationItem
	{
		foreach ($this->getPublishedToDaoNavigationItem($lang) as $v) {
			if ($v->isHomepage)
				return $v;
		}

		return null;
	}

	public function getHomepageForRouter(?string $lang = null): array
	{
		$hp = $this->getHomepage($lang);
		if ($hp) {
			$component = $this->itemsCollector->getItemById($hp->componentType);

			return $component->routerIn($hp, ['getHomepage' => 1]) + ['activeNavigation' => $hp];
		}

		return [];
	}

	public function getIdByUrl($url)
	{
		try {
			$locale = $this->translator->getLocale();
			$key    = self::CACHE_NAMESPACE . '/idByUrl/' . $url['path'] . '/' . $locale;

			if (!$url['locale'])
				$url['locale'] = $locale;

			if (!isset($this->cIdByUrl[$key]) || $this->cIdByUrl[$key] === null) {
				$navigation               = null;
				$navigations              = $this->getPublishedToDaoNavigationItem($locale);
				$navigationsByLink        = [];
				$implodedPartsForRedirect = [];
				$pathParts                = explode('/', $url['path']);

				if (NavigationConfig::load('showDefaultLangInUrl', false) && $url['locale'] === 'cs') {
					if (NavigationConfig::load('useCZinUrl') !== true && $pathParts[0] === 'cz') {
						$url['path'] = 'cs/' . $url['path'];
						array_unshift($pathParts, 'cs');
					} else if ($pathParts[0] !== 'cz') {
						$url['path'] = 'cz/' . $url['path'];
						array_unshift($pathParts, 'cz');
					}
				}

				$tmpPath      = $url['path'];
				$tmpPathParts = $pathParts;
				$return       = [];
				$i            = 0;

				// Vyparsujeme paginator
				$returnPaginator = [];
				$paginatorIndex  = array_search($this->translator->translate('default.urlPart.page'), $pathParts);
				if ($paginatorIndex !== false) {
					$returnPaginator['list-page'] = (int) $pathParts[$paginatorIndex + 1] ?? 1;
					$returnPaginator['do']        = 'list-paginator';

					unset($pathParts[$paginatorIndex]);
					unset($pathParts[$paginatorIndex + 1]);
					$pathParts = array_values($pathParts);

					$url['path']  = implode('/', $pathParts);
					$tmpPath      = $url['path'];
					$tmpPathParts = $pathParts;
				}

				if ($this->translator->getDefaultLocale() !== $locale || NavigationConfig::load('showDefaultLangInUrl')) {
					$localeTmp     = $locale;
					$site          = $this->navigationsHelper->getSitesService()->getCurrentSite();
					$defaultDomain = $site->getDefaultDomain();
					$currDomain    = $site->getDomains()[$locale];
					if ($defaultDomain->domain !== $currDomain->domain || $currDomain->isDefault === 1)
						$localeTmp = '';

					if ($localeTmp === 'cs' && NavigationConfig::load('useCZinUrl') === true)
						$localeTmp = 'cz';

					if ($localeTmp)
						array_unshift($tmpPathParts, $localeTmp);
					$tmpPath = ($localeTmp ? ($localeTmp . '/') : '') . $tmpPath;
				}

				foreach ($navigations as $k => $nav) {
					if (in_array($nav->componentType, ['navigation.alias', 'navigation.customLink']) || !$nav->link)
						continue;

					if (ltrim($nav->link, '/') === $url['path']) {
						$navigation = $nav;
						break;
					}

					$navigationsByLink[trim($nav->link, '/')] = $k;
				}

				// Vyhledání navigace která odpovídá url. Obsah za lomítkem se postupně odebírá
				while (!$navigation && $i < 15 && !empty($tmpPathParts)) {
					$implodedPartsForRedirect[] = $tmpPath;
					$i++;

					foreach ($navigationsByLink as $link => $k) {
						if ($tmpPath === $locale)
							continue;

						if ($link == $tmpPath) {
							$navigation = $navigations[$k];
							break 2;
						}
					}

					array_pop($tmpPathParts);
					$tmpPath = implode('/', $tmpPathParts);
				}

				// Pokud navigace existuje, tak vytvoříme data, jinak zavoláme událost
				if ($navigation) {
					$out = $this->getIdByUrlRouterIn($navigation, $url);

					if (is_array($out))
						$return += $out + ['activeNavigation' => $navigation];
				}

				// Pokud data neexistujou, tak zkusíme vytáhnout data z homepage
				if (empty($return)) {
					$hp = $this->getHomepage($url['locale']);

					if ($hp) {
						$out = $this->getIdByUrlRouterIn($hp, $url);

						if (is_array($out)) {
							$navigation = $hp;
							$return     += $out + ['activeNavigation' => $hp];
						}
					}
				}

				if (empty($return) || !isset($return['presenter'])) {
					$this->eventDispatcher->dispatch(new RouteInEvent(null, $url, $return), Navigations::class . '::routeInNotFound');

					if (isset($return['activeNavigation']))
						$navigation = $return['activeNavigation'];
				}

				// Pokud se navigace nenalezne, tak se podíváme do přesměrání a případně nastavíme navigace nebo rovnou přesměrujeme
				if (!$navigation || !isset($return['presenter'])) {
					foreach ($implodedPartsForRedirect as $v) {
						if (Strings::startsWith($v, $locale) && strlen($v) > 3)
							$implodedPartsForRedirect[] = substr($v, strlen($locale) + 1);
					}

					foreach ($implodedPartsForRedirect as $k => $v) {
						$v                            = str_replace(' ', '%20', $v);
						$implodedPartsForRedirect[$k] = $v;
						$implodedPartsForRedirect[]   = $v . '/';
					}

					usort($implodedPartsForRedirect, fn($a, $b) => strlen($b) <=> strlen($a));

					$redirects = $this->redirectsService->getEr()->createQueryBuilder('r')
						->select('r.to, r.from, r.package, r.relationKey, r.relationValue, r.redirectCode')
						->where('r.from IN (:from)')
						->setParameter('from', $implodedPartsForRedirect)
						->getQuery()->getArrayResult();
					$firstRed  = null;
					$redK      = null;

					// Najduti nejpodobnejsi url
					foreach ($redirects as $k => $red) {
						$key = array_search($red['from'], $implodedPartsForRedirect);
						if (!$firstRed || $key < $firstRed) {
							$firstRed = $key;
							$redK     = $k;
						}
					}

					if (isset($redirects[$redK])) {
						$redirects = $redirects[$redK];

						if ($redirects['relationValue']) {
							$navigation = $navigations[$redirects['relationValue']];
						} else {
							header('Location: ' . $this->navigationsHelper->getHttpRequest()->getUrl()->getBaseUrl() . ltrim($redirects['to'], '/'), true, (int) $redirects['redirectCode']);
							exit;
						}
					}
				}

				// Pokud data stále nejsou tak stránka neexistuje
				if (empty($return))
					$this->cIdByUrl[$key] = false;
				else
					$this->cIdByUrl[$key] = $return + $returnPaginator;
			}

			return $this->cIdByUrl[$key] ?: null;
		} catch (\Exception $e) {
			return null;
		}
	}

	protected function getIdByUrlRouterIn(DaoNavigationItem $navigation, array $params)
	{
		$component = $this->itemsCollector->getItemById($navigation->componentType);

		return $component ? $component->routerIn($navigation, $params) : null;
	}

	public function getUrlById(&$params)
	{
		$return = null;
		try {
			$domain = $params['domain'];
			unset($params['domain']);

			$tmp = $params;
			if ($tmp['activeNavigation']) {
				$tmp['activeNavigation'] = $tmp['activeNavigation']->id;
			}

			if (!isset($params['locale']))
				$params['locale'] = $this->translator->getLocale() ?: Parameters::load('translation.default', 'cs');

			foreach ($tmp as $v)
				if (is_object($v))
					unset($tmp[$v]);

			$key = 'urlById/' . $this->getCurrentSiteIdent() . '/' . serialize($tmp);
			if ($this->cUrlById[$key] === null) {
				$dep = $this->cacheDep;

				if (isset($params['activeNavigation'])) {
					$navType = $params['activeNavigation']->componentType;

					if ($navType == 'navigation.alias') {
						$tmp = $params['activeNavigation']->getParam('aliasReference');

						if ($tmp) {
							$params['activeNavigation'] = $tmp;
							$navType                    = $tmp->componentType;
						}
					}
					$component = $this->itemsCollector->getItemById($navType);

					if ($component) {
						$preAlias = $this->getPreAlias($params['activeNavigation']);

						if ($params['activeNavigation']->isFullLinkFilled)
							$return = $params['activeNavigation']->link;
						else
							$return = $preAlias . $component->routerOut($params['activeNavigation'], $params);
					}
				}

				if (!$return) {
					$groups = $this->itemsCollector->findItemByPresenter($params['presenter'], $params['action']);

					if (!$groups) {
						if ($params['presenter']) {
							preg_match('/Override$/', $params['presenter'], $matches, PREG_OFFSET_CAPTURE, 0);

							if (isset($matches[0]))
								$params['presenter'] = substr($params['presenter'], 0, $matches[0][1]);
						}

						$groups = $this->itemsCollector->findItemByPresenter($params['presenter'], $params['action']);
					}

					if (!$groups)
						return null;

					ksort($groups);

					foreach ($groups as $components) {
						/** @var INavigationItem[]|BaseItem[] $components */
						foreach ($components as $component) {
							foreach ($this->getPublishedByComponent($component->getId(), $params['locale']) as $navigation) {
								$return = $component->routerOut($navigation, $params);

								if ($return === '')
									$return = null;
								else if ($return !== null && $return !== false) {
									$component->updateCacheDep($dep, $params);
									$return = $this->getPreAlias($navigation) . $return;
									break 3;
								}
							}
						}
					}
				}
				if (isset($params['list-page'])) {
					if ($params['list-page'] > 1) {
						$return .= '/' . $this->translator->translate('default.urlPart.page') . '/' . $params['list-page'];
					}

					unset($params['list-page']);
					unset($params['do']);
				}

				if (isset($params['activeNavigation']) && !$params['activeNavigation']->link)
					$params['activeNavigation']->link = $return;

				$this->cUrlById[$key] = ['return' => $return, 'params' => $params];
			}

			$params = $this->cUrlById[$key]['params'];
			$return = $this->cUrlById[$key]['return'];

			$url = ($domain ? 'https://' . $domain : '') . $return;

			if (NavigationConfig::load('useEndSlash'))
				$url = rtrim($url, '/') . '/';

			return $url;
		} catch (\Exception $e) {
		}

		return null;
	}

	/**
	 * @param DaoNavigationItem $navigation
	 *
	 * @return string
	 */
	public function getPreAlias(DaoNavigationItem $navigation)
	{
		if ($navigation->componentType === CustomLink::TYPE) {
			return '';
		}

		if ($navigation->isFullLinkFilled) {
			$url = '';
		} else {
			$arr = [];
			$t   = $navigation->parent;

			while ($t) {
				$arr[] = $t->alias;
				$t     = $t->parent;
			}

			$url = $arr ? '/' . implode('/', array_reverse($arr)) : '';
		}
		$this->setLocaleUrlPrefix($url, $navigation);

		return $url;
	}

	public function setLocaleUrlPrefix(string &$url, ?DaoNavigationItem $navigation = null): void
	{
		$showDefaultLangInUrl = NavigationConfig::load('showDefaultLangInUrl', false);
		$domain               = $this->navigationsHelper->getSitesService()->getCurrentSite()->getDomains()[$navigation->getLang()] ?? null;

		if ($domain && $domain->isDefault === 1 && $domain->getLang() === $navigation->getLang() && !NavigationConfig::load('showDefaultLangInUrl', false))
			return;

		if ($navigation->getLang() && ($this->translator->getDefaultLocale() != $navigation->getLang() || $showDefaultLangInUrl)) {
			$lang = $navigation->getLang();
			$site = $this->navigationsHelper->getSitesService()->getCurrentSite();

			$defaultDomain = $site->getDefaultDomain();
			$currDomain    = $site->getDomains()[$lang];
			if ($defaultDomain->domain !== $currDomain->domain)
				$lang = '';

			if ($lang === 'cs' && NavigationConfig::load('useCZinUrl') === true)
				$lang = 'cz';

			$url = ($lang ? ('/' . $lang) : '') . $url;
		}
	}

	/*******************************************************************************************************************
	 * ===========================  Cache
	 */
}
