<?php declare(strict_types = 1);

namespace Navigations\AdminModule\Components;

use Core\Model\Lang\Langs;
use Core\Model\UI\BaseControl;
use Navigations\AdminModule\Model\Navigations;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\NavigationsGroups;
use Nette\Application\ForbiddenRequestException;
use Nette\Utils\Html;

class NavigationsGrid extends BaseControl
{
	/** @var NavigationsGroups */
	protected $navigationsGroupsService;

	/** @var Navigations */
	protected $navigationsService;

	/** @var Langs */
	protected $langsService;

	public function __construct(Navigations $navigations, NavigationsGroups $groups, Langs $langs)
	{
		$this->navigationsService       = $navigations;
		$this->navigationsGroupsService = $groups;
		$this->langsService             = $langs;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		$result    = $this->navigationsService->removeNavigation($id);
		if (is_string($result))
			$presenter->flashMessageDanger($result);
		elseif ($result === true)
			$presenter->flashMessageSuccess('default.removed');
		else
			$presenter->flashMessageDanger('default.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleFixTreeStructure()
	{
		set_time_limit(0);
		if (!$this->getPresenter()->getUser()->getIdentity()->isInRole('Super admin'))
			throw new ForbiddenRequestException();
		$this->template->devResponse = $this->navigationsService->fixTreeStructure();
		$this->redrawControl('devResponse');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();

		$qb = $this->navigationsService->getEr()->createQueryBuilder('n')
			->join('n.group', 'g')
			->where('n.lvl > 0')
			->addOrderBy('g.id', 'ASC')->addOrderBy('n.lft', 'ASC');

		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('navigationsGrid:gridSortableRow!');

		// Columns
		$grid->addColumnLink('title', 'default.title', 'Default:edit')->setRenderer(function($row) {
			/** @var Navigation $row */
			$title = ' ' . $row->title;
			for ($i = 1; $i < $row->getLvl(); $i++)
				$title = '---' . $title;

			return Html::el('a', ['href' => $this->getPresenter()->link('Default:edit', [$row->getId()])])->setText($title);
		});
		$grid->addColumnText('alias', 'default.alias');
		$grid->addColumnText('group', 'default.group', 'group.title');
		$grid->addColumnText('lang', 'default.lang');
		$grid->addColumnStatus('isHomepage', 'navigations.navigation.isHomepage')->setAlign('center')
			->addOption(1, 'navigations.navigation.setAsHomepage')->setIcon('star')->setShowTitle(false)->endOption()
			->addOption(0, '')->setIcon('star far')->setClass('btn-default')->setClassInDropdown('hidden')->endOption()
			->onChange[] = [$this, 'gridHomepageChange'];
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Filter
		$groups = ['' => ''];
		foreach ($this->navigationsGroupsService->getAll() as $group)
			$groups[$group->getId()] = $group->title;

		$langs = ['' => ''];
		foreach ($this->langsService->getLangs() as $lang) {
			$langs[$lang->getTag()] = $lang->getTitle();
		}

		$grid->addFilterText('title', 'default.title');
		$grid->addFilterText('alias', 'default.alias');
		$grid->addFilterSelect('group', 'default.group', $groups, 'g.id');
		$grid->addFilterSelect('lang', 'default.lang', $langs, 'n.lang');

		// Actions
		$grid->addAction('edit', '', 'Default:edit')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')
			->setConfirm('default.reallyDelete');

		// Columns prototype
		$grid->getColumn('isPublished')->getElementPrototype('th')->class[] = 'w1';
		$grid->getColumn('isHomepage')->getElementPrototype('th')->class[]  = 'w1';

		$grid->setRowCallback(function($row, $tr) {
			$groupId = ($row->group ? $row->group->getId() : 0) . '-' . ($row->parent ? $row->parent->getId() : '0') . '-' . $row->getLvl();
			/** @var Html $tr */
			$tr->addClass('group_' . $groupId);
			$tr->addAttributes(['data-group-tree' => $groupId]);

			return $tr;
		});

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->navigationsService->setPublish($id, $newStatus))
			$presenter->flashMessageSuccess('default.publishChanged');
		else
			$presenter->flashMessageDanger('default.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function gridHomepageChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($newStatus == 1 && $this->navigationsService->setHomepage($id))
			$presenter->flashMessageSuccess('navigations.navigationForm.homepageChanged');
		else
			$presenter->flashMessageDanger('navigations.navigationForm.homepageChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleGridSortableRow()
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();
		$type      = $request->getPost('type');

		switch ($type) {
			case 'default':
				$id       = $request->getPost('id');
				$position = $request->getPost('position');

				if ($id != null && $position != null && $this->navigationsService->setPosition($id, $position))
					$presenter->flashMessageSuccess('default.positionChanged');
				else
					$presenter->flashMessageDanger('default.positionChangeFailed');
				break;
			case 'tree':
				$id   = $request->getPost('id');
				$move = $request->getPost('move');
				$er   = $this->navigationsService->getEr();
				$nav  = $this->navigationsService->get($id);

				if ($id && $move) {
					if ($move < 0)
						$er->moveUp($nav, abs($move));
					else if ($move > 0)
						$er->moveDown($nav, $move);
					$presenter->flashMessageSuccess('default.positionChanged');
				} else
					$presenter->flashMessageDanger('default.positionChangeFailed');

				$this->em->flush();
				$this->navigationsService->cleanCache();

				break;
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}
}
