<?php declare(strict_types = 1);

namespace Navigations\AdminModule\Model;

use Core\Model\Entities\Repository\NestedTreeRepository;
use Core\Model\Helpers\BaseEntityService;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Entities\NavigationGroup;
use Nette\Caching\Cache;

/**
 * Class Navigations
 * @package Navigations\AdminModule\Model
 *
 * @method Navigation|object|null getReference($id)
 * @method Navigation[]|null getAll()
 * @method Navigation|null get($id)
 * @method NestedTreeRepository getEr()
 */
class Navigations extends BaseEntityService
{
	protected $entityClass = Navigation::class;

	const CACHE_NAMESPACE = 'navigations';

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	public function cleanCache()
	{
		$this->getCache()->clean([Cache::TAGS => [self::CACHE_NAMESPACE]]);
	}

	/**
	 * @param $id
	 * @param $state
	 *
	 * @return bool
	 * @throws \Exception
	 */
	public function setPublish($id, $state)
	{
		if ($nav = $this->get($id)) {
			$nav->isPublished = $state;
			$this->em->persist($nav);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/**
	 * @param $id
	 *
	 * @return bool
	 * @throws \Exception
	 */
	public function setHomepage($id)
	{
		if ($nav = $this->get($id)) {
			$q = $this->getEr()->createQueryBuilder('n')->update(Navigation::class, 'n')
				->set('n.isHomepage', '0')
				->where('n.isHomepage != 0');
			if ($nav->getLang())
				$q->andWhere('n.lang = :lang')->setParameter('lang', $nav->getLang());
			else
				$q->andWhere('n.lang IS NULL');
			$q->getQuery()->execute();

			$nav->isHomepage = 1;
			$this->em->persist($nav);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/**
	 * @param $id
	 *
	 * @return bool|string
	 * @throws \Exception
	 */
	public function removeNavigation($id)
	{
		if ($nav = $this->get($id)) {
			if ($nav->getChildren()->count())
				return 'navigations.navigation.removeFailedHasChilds';
			$this->em->remove($nav);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/**
	 * @param $id
	 * @param $position
	 *
	 * @return bool
	 * @throws \Exception
	 */
	public function setPosition($id, $position)
	{
		if ($item = $this->getReference($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/**
	 * @param $alias
	 * @param $locale
	 *
	 * @return Navigation|null|object
	 */
	public function findByAlias($alias, $locale)
	{
		$navigations = $this->getEr()->createQueryBuilder('n')
			->andWhere('n.alias = :alias')->setParameter('alias', $alias);

		if ($locale)
			$navigations->andWhere('n.lang = :locale')->setParameter('locale', $locale);
		else
			$navigations->andWhere('n.lang IS NULL');

		$navigations = $navigations->getQuery()->getResult();

		$navigation = null;

		foreach ($navigations as $v)
			if ($v->lang == $locale) {
				$navigation = $v;
				break;
			}

		if (!$navigation)
			$navigation = array_values($navigations)[0];

		return $navigation;
	}

	/**
	 * @return array
	 * @throws \Exception
	 */
	public function fixTreeStructure()
	{
		$result = ['status' => 'error'];

		$groups  = $this->em->getRepository(NavigationGroup::class)->createQueryBuilder('ng')->select('ng.title, ng.id')->getQuery()->getArrayResult();
		$parents = [];
		/** @var Navigation[] $navs */
		$navs = $this->getEr()->createQueryBuilder('n')->orderBy('n.parent')->addOrderBy('n.position', 'DESC')->getQuery()->getResult();
		/** @var Navigation[] $roots */
		$roots = [];

		// Vytvoření dočasného rootu
		$tmp     = $this->em->getRepository(NavigationGroup::class)->findOneBy([]);
		$tmpRoot = new Navigation('ROOT - TMP', $this->em->getReference(NavigationGroup::class, $tmp->getId()), '', 'navigation.customLink', 0);
		$this->em->persist($tmpRoot);
		$this->em->flush();

		// Získání všech rootů
		foreach ($groups as $g) {
			$root = $this->getEr()->findOneBy(['title' => 'ROOT - ' . $g['title']]);
			if (!$root) {
				$root = new Navigation('ROOT - ' . $g['title'], $this->em->getReference(NavigationGroup::class, $g['id']), '', 'navigation.customLink', 0);
				$this->em->persist($root);
				$this->em->flush();
			}
			$this->getEr()->createQueryBuilder('n')->update()
				->set('n.root', ':root')
				->where('n.id = :id')
				->getQuery()->execute(['root' => $root->getId(), 'id' => $root->getId()]);
			$roots[$g['id']] = $root;
		}

		// Získání všech navigací bez rootu
		foreach ($navs as $k => $n) {
			if ($roots[$n->group->getId()]->getId() == $n->getId() || strpos($n->title, 'ROOT - ') === 0) {
				unset($navs[$k]);
				continue;
			}

			if ($n->getParent())
				$parents[$n->getId()] = $n->getParent();
		}

		// Vložení do dočasného rootu
		foreach ($navs as $n) {
			$this->getEr()->persistAsFirstChildOf($n, $tmpRoot);
			$this->em->flush();
		}

		$this->getEr()->verify();

		// Vložení do správných rootů
		foreach ($navs as $n) {
			$this->getEr()->persistAsFirstChildOf($n, $roots[$n->group->getId()]);
			$this->em->flush();
		}

		// Nastavení nadřazených
		foreach ($navs as $n) {
			if (isset($parents[$n->getId()])) {
				$this->getEr()->persistAsFirstChildOf($n, $parents[$n->getId()]);
				$this->em->flush();
			}
		}

		$errors = $this->getEr()->verify();
		$this->em->flush();

		$this->em->remove($tmpRoot);
		$tmpRoots = $this->getEr()->findBy(['title' => 'ROOT - TMP']);
		if ($tmpRoots)
			$this->em->remove($tmpRoots);
		$this->em->flush();

		if (is_array($errors))
			$result['errors'] = $errors;
		else
			$result['status'] = 'ok';

		return $result;
	}
}
