<?php declare(strict_types = 1);

namespace Navigations\Model\Entities;

use Core\Model\Entities\Redirect;
use Core\AdminModule\Model\Redirects;
use function Couchbase\basicDecoderV1;
use Doctrine;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Kdyby\Events\Subscriber;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Nette\Caching\IStorage;
use Nette\Http\Session;
use Nette\SmartObject;

class NavigationListener implements Subscriber
{
	use SmartObject;

	/** @var Cache */
	private $cache;

	/** @var Cache */
	private $baseCache;

	/** @var Session */
	private $session;

	/** @var Redirects */
	protected $redirectsService;

	/** @var Navigations */
	protected $navigationsService;

	protected $fields = false;

	public function __construct(IStorage $cacheStorage, Session $session, Redirects $redirects, Navigations $navigations)
	{
		$this->session            = $session;
		$this->cache              = new Cache($cacheStorage, Navigations::CACHE_NAMESPACE);
		$this->baseCache          = new Cache($cacheStorage, 'base');
		$this->redirectsService   = $redirects;
		$this->navigationsService = $navigations;
	}

	public function getSubscribedEvents()
	{
		return [];
	}

	/**
	 * @ORM\PostPersist
	 * @ORM\PostUpdate
	 * @ORM\PostRemove
	 *
	 * @param Navigation         $navigation
	 * @param LifecycleEventArgs $event
	 */
	public function postHandler(Navigation $navigation, LifecycleEventArgs $event)
	{
		$this->cache->clean([
			Cache::TAGS => [Navigations::CACHE_NAMESPACE],
		]);
		$this->baseCache->clean([
			Cache::TAGS => ['navigation'],
		]);
	}

	/**
	 * @ORM\PostUpdate
	 *
	 * @param Navigation         $navigation
	 * @param LifecycleEventArgs $event
	 *
	 * @throws Doctrine\ORM\ORMException
	 */
	public function postUpdateHandler(Navigation $navigation, LifecycleEventArgs $event)
	{
		$em        = $event->getEntityManager();
		$changeSet = $em->getUnitOfWork()->getEntityChangeSet($navigation);

		if (isset($changeSet['alias']) && !$navigation->isHomepage) {
			list($from, $to) = $changeSet['alias'];
			$path = $this->navigationsService->getEr()->getPath($navigation);
			array_shift($path);
			array_pop($path);

			if ($path) {
				$tFrom = [];
				$tTo   = [];
				foreach ($path as $p) {
					$tFrom[] = $p->alias;
					$tTo[]   = $p->alias;
				}
				$tFrom[] = $from;
				$tTo[]   = $to;
				$from    = implode('/', $tFrom);
				$to      = implode('/', $tTo);
			}

			$fromExist = $this->redirectsService->getEr()->findOneBy(['from'        => $to, 'package' => 'Navigations',
			                                                          'relationKey' => 'Navigation']);
			if ($fromExist)
				$em->remove($fromExist);

			$toExist = $this->redirectsService->getEr()->findBy(['to'          => $from, 'package' => 'Navigations',
			                                                     'relationKey' => 'Navigation']);
			foreach ($toExist as $red) {
				$red->to = $to;
				$em->persist($red);
			}

			// TODO přesunout do služby?
			$redirect                = new Redirect('Navigation redirect - ' . $from . ' -> ' . $to, $from, $to);
			$redirect->isLocked      = 1;
			$redirect->package       = reset(explode('\\', __NAMESPACE__));
			$redirect->relationKey   = 'Navigation';
			$redirect->relationValue = $navigation->getId();

			$em->persist($redirect);
			$em->flush();
		}
	}
}
