<?php declare(strict_types = 1);

namespace Navigations\Model;

use Contributte\EventDispatcher\EventDispatcher;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Helpers\BaseEntityService;
use Core\FrontModule\Model\Redirects;
use Core\Model\Lang\DefaultLang;
use EshopCatalog\Model\Event\RouteInEvent;
use Gedmo\Tree\Entity\Repository\NestedTreeRepository;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Entities\NavigationGroup;
use Navigations\Model\Providers\INavigationItem;
use Nette\Caching\Cache;
use Nette\Localization\ITranslator;

/**
 * Class Navigations
 * @package Navigations\Model
 *
 * @method Navigation|object|null getReference($id)
 * @method Navigation|null get($id)
 * @method NestedTreeRepository getEr()
 */
class Navigations extends BaseEntityService
{

	const CACHE_NAMESPACE = 'navigations';

	protected $entityClass = Navigation::class;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => [self::CACHE_NAMESPACE],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	/** @var ITranslator */
	public $translator;

	/** @var ItemsCollector @inject */
	public $itemsCollector;

	/** @var Redirects @inject */
	public $redirectsService;

	/** @var DefaultLang @inject */
	public $defaultLang;

	/** @var EventDispatcher @inject */
	public $eventDispatcher;

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	/**
	 * @param Navigation $navigation
	 *
	 * @return Navigation[]
	 */
	public function getPath($navigation)
	{
		if (!$navigation)
			return [];

		$key = self::CACHE_NAMESPACE . '/path/' . $navigation->getId();

		return $this->getCache()->load($key, function(&$dep) use ($navigation) {
			$dep = $this->cacheDep;

			return $this->getEr()->getPath($navigation);
		});
	}

	/**
	 * @return Navigation|null|object
	 */
	public function getHomepageEntity()
	{
		$key = self::CACHE_NAMESPACE . '/homepageEntity.' . $this->defaultLang->locale;

		return $this->getCache()->load($key, function(&$dep) {
			$dep = $this->cacheDep;

			return $this->getEr()->createQueryBuilder('n')
				->where('n.isHomepage = 1')->andWhere('n.lvl > 0')
				->andWhere('n.lang IS NULL OR n.lang = :locale')->setParameter('locale', $this->defaultLang->locale)
				->getQuery()->getOneOrNullResult();
		});
	}

	/**
	 * @return null|Navigation[]
	 */
	public function getAll()
	{
		$key = self::CACHE_NAMESPACE . '/all';

		return $this->getCache()->load($key, function(&$dep) {
			$dep = $this->cacheDep;

			return $this->getEr()->createQueryBuilder('n')->where('n.lvl > 0')->getQuery()->getResult();
		});
	}

	/**
	 * @param int $id
	 *
	 * @return null|Navigation[]
	 */
	public function getByGroup($id)
	{
		$key = self::CACHE_NAMESPACE . '/group/' . $id;

		return $this->getCache()->load($key, function(&$dep) use ($id) {
			$dep = $this->cacheDep;

			return $this->getEr()->createQueryBuilder('n')
				->where('n.group = :group')->andWhere('n.lvl > 0')
				->setParameter('group', $id)
				->getQuery()->getResult();
		});
	}

	/**
	 * @return Navigation[]|null
	 */
	public function getPublished()
	{
		$key = self::CACHE_NAMESPACE . '/published/l-' . $this->defaultLang->locale;

		return $this->getCache()->load($key, function(&$dep) {
			$dep = $this->cacheDep;

			$navigations = $this->getEr()->createQueryBuilder('n')->where('n.isPublished = 1')
				->andWhere('n.lvl > 0')
				->andWhere('n.lang IS NULL OR n.lang = :locale')->setParameter('locale', $this->defaultLang->locale)
				->leftJoin('n.parent', 'p')->addSelect('p')
				->leftJoin('n.children', 'child')->addSelect('child')
				->orderBy('n.root')->addOrderBy('n.lft')->getQuery()->getResult();

			if ($navigations) {
				$gIds = [];
				foreach ($navigations as $nav) {
					$gIds[$nav->group->getId()] = $nav->group->getId();
				}
				$this->em->getRepository(NavigationGroup::class)->createQueryBuilder('g')
					->andWhere(is_array($gIds) ? 'g.id IN (:gid)' : 'g.id = :gid')->setParameter('gid', $gIds)
					->getQuery()->getResult();
			}

			return $navigations;
		});
	}

	/**
	 * Vytvoří strukturu navigace složenou z DaoNavigationItem pro komponentu navigace
	 *
	 * @return array
	 */
	public function getPublishedToDaoNavigationItem()
	{
		$key = self::CACHE_NAMESPACE . '/publishedDaoNavigationItem/l-' . $this->defaultLang->locale;

		return $this->getCache()->load($key, function(&$dep) {
			$dep  = $this->cacheDep;
			$data = [];

			foreach ($this->getPublished() as $nav) {
				if ($nav->getLvl() == 1)
					$data[$nav->group->type][] = $this->fillDao($nav);
			}

			return $data;
		});
	}

	public function fillDao($item, $dao = null)
	{
		/** @var Navigation $item */
		if (!$dao) {
			$dao = new DaoNavigationItem($item);
		}

		$dao->set('componentType', $item->componentType);
		$dao->set('componentParams', $item->componentParams);
		$dao->set('id', $item->getId());
		$dao->set('link', $this->generateLink($item));
		$dao->set('class', trim('item-' . $item->getId() . ($item->getParam('linkClass') ?: '')));

		foreach ($item->getChildren()->toArray() as $child) {
			if (!$child->isPublished)
				continue;
			$daoC = new DaoNavigationItem($child, $dao);
			$daoC = $this->fillDao($child, $daoC);

			if ($daoC) {
				$dao->addChild($daoC);
			}
		}

		return $dao;
	}

	/**
	 * @param $id
	 *
	 * @return array|mixed
	 */
	public function getPublishedByGroup($id)
	{
		$key = self::CACHE_NAMESPACE . '/publishedByGroup/' . $id . '/l-' . $this->defaultLang->locale;

		return $this->getCache()->load($key, function(&$dep) use ($id) {
			$dep = $this->cacheDep;

			return $this->em->createQueryBuilder()->select('n')->from(Navigation::class, 'n')
				->where('n.group = :group')->andWhere('n.isPublished = 1')->setParameter('group', $id)
				->andWhere('n.lvl > 0')
				->andWhere('n.lang IS null OR n.lang = :locale')->setParameter('locale', $this->defaultLang->locale)
				->orderBy('n.root')->addOrderBy('n.lft')->getQuery()->getResult();
		});
	}

	/**
	 * @param string $type
	 * @param bool   $toDao
	 *
	 * @return array|null
	 */
	public function getPublishedByGroupType($type, bool $toDao = false)
	{
		$key = 'navByGroupType/' . $type . '/l-' . $this->defaultLang->locale . ($toDao ? '/d' : '');

		return $this->getCache()->load($key, function(&$dep) use ($type, $toDao) {
			$dep = $this->cacheDep;

			$q = $this->em->createQueryBuilder()->select('n')->from(Navigation::class, 'n')
				->join('n.group', 'g')
				->where('g.type = :type')->andWhere('n.isPublished = 1')->setParameter('type', $type)
				->andWhere('n.lvl > 0')
				->andWhere('n.lang IS null OR n.lang = :locale')->setParameter('locale', $this->defaultLang->locale)
				->orderBy('n.root')->addOrderBy('n.lft')->getQuery();

			if ($toDao) {
				$data = [];
				foreach ($q->getResult() as $nav) {
					$data[] = $this->fillDao($nav);
				}

				return $data;
			} else {
				return $q->getArrayResult();
			}
		});
	}

	/**
	 * @param string $component
	 *
	 * @return mixed
	 */
	public function getPublishedByComponent($component)
	{
		$key = self::CACHE_NAMESPACE . '/publishedByComponent/' . $component . '/l-' . $this->defaultLang->locale;

		return $this->getCache()->load($key, function(&$dep) use ($component) {
			$dep = $this->cacheDep;

			return $this->getEr()->createQueryBuilder('n')->where('n.isPublished = 1')
				->andWhere('n.lvl > 0')
				->andWhere('n.componentType = :component')->setParameter('component', $component)
				->andWhere('n.lang IS null OR n.lang = :locale')->setParameter('locale', $this->defaultLang->locale)
				->getQuery()->getResult();
		});
	}

	/**
	 * @param $id
	 *
	 * @return null|object|Navigation
	 */
	public function getNavigation($id)
	{
		$key = self::CACHE_NAMESPACE . '/nav/' . $id;

		return $this->getCache()->load($key, function(&$dep) use ($id) {
			$dep = $this->cacheDep;
			$nav = $this->getEr()->find($id);

			if ($nav->children)
				$nav->children->toArray();

			return $nav;
		});
	}

	/**
	 * @param $alias
	 *
	 * @return Navigation|null|object
	 */
	public function findByAlias($alias)
	{
		$oAlias = $alias;
		$locale = is_array($alias) && isset($alias['locale']) ? $alias['locale'] : null;
		if (!is_string($alias))
			$alias = is_array($alias) && !empty($alias) && isset($alias['path']) ? $alias['path'] : null;

		if (!$alias)
			return null;

		if (is_string($alias))
			$alias = [$alias];

		$navigation = null;
		$a          = end($alias);

		// Projdutí všech alisů odzadu
		$navigations = $this->getEr()->createQueryBuilder('n')
			->andWhere('n.alias = :alias')->setParameter('alias', $a)
			->andWhere('n.lvl > 0')
			->andWhere('n.lang IS NULL OR n.lang = :locale')->setParameter('locale', $locale)
			->andWhere('n.isPublished = 1')
			->getQuery()->getResult();

		foreach ($navigations as $n) {
			$path  = $this->getEr()->getPath($n);
			$match = true;

			if ($path[0]->getLvl() == 0)
				array_shift($path);

			// Vyhledání správné cesty podle nadřezených
			foreach ($alias as $k => $v) {
				if ($v != $path[$k]->alias) {
					$match = false;
					break;
				}
			}

			if ($match) {
				$navigation = $n;
				break;
			}

			// Vyhledání správné cesty podle nadřezených 2
			foreach ($path as $k => $p) {
				if ($p->alias != $alias[$k]) {
					$match = false;
					break;
				}
			}

			if ($match) {
				$navigation = $n;
				break;
			}
		}

		// Vyhledání v přesměrování
		if (!$navigation) {
			$redirect = $this->redirectsService->getEr()->findOneBy(['from'        => implode('/', $alias),
			                                                         'package'     => 'Navigations',
			                                                         'relationKey' => 'Navigation']);
			if ($redirect && $redirect->relationValue !== null)
				$navigation = $this->get($redirect->relationValue);
			else {
				array_pop($oAlias['path']);

				if ($oAlias['path']) {
					$navigation = $this->findByAlias($oAlias);
				}
			}
		}

		return $navigation;
	}

	/**
	 * Zpracování navigace a získání URL
	 *
	 * @param Navigation $navigation
	 *
	 * @return array|bool
	 */
	public function generateLink(Navigation $navigation)
	{
		$t = ['activeNavigation' => $navigation];

		return $this->getUrlById($t);
	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function getHomepage($locale = null, $params = [])
	{
		try {
			$locale = $locale ?? $this->defaultLang->locale;
			$key    = self::CACHE_NAMESPACE . '/route/hp/' . $locale;

			return $this->getCache()->load($key, function(&$dep) use ($key, $locale, $params) {
				$dep = $this->cacheDep;

				$navigation = $this->getEr()->createQueryBuilder('n')
					->where('n.isHomepage = 1')->andWhere('n.isPublished = 1')
					->andWhere('n.lvl > 0')
					->andWhere('n.lang IS NULL OR n.lang = :locale')->setParameter('locale', $locale)
					->getQuery()->getOneOrNullResult();

				if (!$navigation || !($component = $this->itemsCollector->getItemById($navigation->componentType)))
					return null;

				$params['getHomepage'] = 1;

				return $component->routerIn($navigation->componentParams, $params) + ['activeNavigation' => $navigation];
			});
		} catch (\Exception $e) {
		}

		return null;
	}

	public function getIdByUrl($url)
	{
		try {
			$return = [];
			$locale = $url['locale'] ?? $this->defaultLang->locale;
			$key    = self::CACHE_NAMESPACE . '/idByUrl/' . serialize($url) . '/l-' . $locale;

			return $this->getCache()->load($key, function(&$dep) use ($url, $return, $locale) {
				$dep = $this->cacheDep;

				$navigation = null;

				if (isset($url['path'])) {
					$tmp = explode('/', $url['path']);
					$t   = $tmp;

					// Upravení pro stránkování u obecné komponenty
					$paginatorIndex = array_search($this->translator->translate('default.urlPart.page'), $t);
					if ($paginatorIndex !== false) {
						$return['list-page'] = $t[$paginatorIndex + 1] ?? 1;
						$return['do']        = 'list-paginator';
						unset($t[$paginatorIndex + 1]);
						unset($t[$paginatorIndex]);
						unset($tmp[$paginatorIndex + 1]);
						unset($tmp[$paginatorIndex]);
					}

					if (!$navigation) {
						$navigation = $this->findByAlias(['path' => $t, 'locale' => $locale]);
					}
					$url['path'] = implode('/', $tmp);
				}

				if ($navigation && $component = $this->itemsCollector->getItemById($navigation->componentType)) {
					$return += $component->routerIn($navigation, $url) + ['activeNavigation' => $navigation];
				} else if (!$navigation) {
					$tmp       = $this->getHomepage($url['locale'], $url);
					$component = $this->itemsCollector->getItemById($tmp['activeNavigation']->componentType);

					if (!$component)
						return null;
					$return += $component->routerIn($tmp['activeNavigation']->componentParams, $url);


					if ($return['id'] == null) {
						$return = [];
						$this->eventDispatcher->dispatch(Navigations::class . '::routeInNotFound', new RouteInEvent(null, $url, $return));

						return empty($return) ? null : $return;
					}

					//					$tmp['activeNavigation']->isHomepage = 0;

					$return += ['activeNavigation' => $tmp['activeNavigation']];
				}

				$component->updateCacheDep($dep, $return);

				return $return;
			});
		} catch (\Exception $e) {
			return null;
		}
	}

	public function getUrlById(&$params)
	{
		$return = null;
		try {
			$tmp = $params;
			if ($tmp['activeNavigation'])
				$tmp['activeNavigation'] = $tmp['activeNavigation']->getId();
			foreach ($tmp as $v)
				if (is_object($v))
					unset($tmp[$v]);

			$key = self::CACHE_NAMESPACE . '/urlById/' . serialize($tmp);

			$data   = $this->getCache()->load($key, function(&$dep) use ($params, $return) {
				$dep = $this->cacheDep;

				if (isset($params['activeNavigation'])) {
					$navType = $params['activeNavigation']->componentType;
					if ($navType == 'navigation.alias') {
						$tmp = $this->getNavigation($params['activeNavigation']->componentParams['navigation']);

						if ($tmp) {
							$params['activeNavigation'] = $tmp;
							$navType                    = $tmp->componentType;
						}
					}
					$component = $this->itemsCollector->getItemById($navType);

					if ($component)
						$return = $this->getPreAlias($params['activeNavigation']) . $component->routerOut($params['activeNavigation'], $params);
				}

				if (!$return) {
					$groups = $this->itemsCollector->findItemByPresenter($params['presenter'], $params['action']);

					if (!$groups) {
						if ($params['presenter']) {
							preg_match('/Override$/', $params['presenter'], $matches, PREG_OFFSET_CAPTURE, 0);

							if (isset($matches[0]))
								$params['presenter'] = substr($params['presenter'], 0, $matches[0][1]);
						}

						$groups = $this->itemsCollector->findItemByPresenter($params['presenter'], $params['action']);
					}

					if (!$groups)
						return null;

					ksort($groups);

					foreach ($groups as $components) {
						/** @var INavigationItem[]|BaseItem[] $components */
						foreach ($components as $component) {
							foreach ($this->getPublishedByComponent($component->getId()) as $navigation) {
								$return = $component->routerOut($navigation, $params);

								if ($return === '')
									$return = null;
								else if ($return !== null && $return !== false) {
									$component->updateCacheDep($dep, $params);
									$return = $this->getPreAlias($navigation) . $return;
									break 3;
								}
							}
						}
					}
				}

				// Upravení url pro stránkování
				if (isset($params['do']) && isset($params['list-page']) && $params['do'] == 'list-paginator') {
					if ($params['list-page'] > 1) {
						$return .= '/' . $this->translator->translate('default.urlPart.page') . '/' . $params['list-page'];
					}

					unset($params['list-page']);
					unset($params['do']);
				}

				if (isset($params['activeNavigation']))
					$params['activeNavigation']->link = $return;

				return ['return' => $return, 'params' => $params];
			});
			$params = $data['params'];

			return $data['return'];
		} catch (\Exception $e) {
		}

		return null;
	}

	/**
	 * @param Navigation $navigation
	 *
	 * @return string
	 */
	public function getPreAlias($navigation)
	{
		$arr = [];
		$t   = $navigation->parent;

		while ($t && $t->parent) {
			$arr[] = $t->alias;
			$t     = $t->parent;
		}

		if ($navigation->getLang() && $this->defaultLang->locale != $navigation->getLang())
			$arr[] = $navigation->getLang();

		return $arr ? '/' . implode('/', array_reverse($arr)) : '';
	}

	/*******************************************************************************************************************
	 * ===========================  Cache
	 */
}
