<?php declare(strict_types = 1);

namespace Navigations\AdminModule\Components;

use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\FormValidateEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Navigations\AdminModule\Model\NavigationsGroups;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Entities\NavigationGroup;
use Navigations\Model\Entities\NavigationGroupText;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;

class NavigationGroupForm extends BaseControl
{
	/** @var int @persistent */
	public $groupId;

	/** @var NavigationGroup */
	public $group;

	/** @var NavigationsGroups @inject */
	public $navigationGroupsService;

	public function render()
	{
		$this->eventDispatcher->dispatch(NavigationGroupForm::class . '::render', new ComponentTemplateEvent($this->template, $this));
		$this->template->render($this->getTemplateFile() ?: __DIR__ . '/NavigationGroupForm.latte');
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('title', 'default.title')->setMaxLength(255)
			->setIsMultilanguage();
		$form->addText('type', 'default.type')->setMaxLength(255)->setRequired();

		$this->eventDispatcher->dispatch(NavigationGroupForm::class . '::createForm',
			new CreateFormEvent($form, $this->getPresenter(false) ? $this->template : null));

		$form->addSaveCancelControl('saveControl');

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$presenter = $this->getPresenter(false);

		$nameFilled = false;
		foreach ($values->title as $v)
			if ($v)
				$nameFilled = true;

		if (!$nameFilled)
			$form['title']->addError('default.formMessages.required');
		if ($form->hasErrors())
			$this->redrawControl('form');

		$this->eventDispatcher->dispatch(NavigationGroupForm::class . '::validateForm',
			new FormValidateEvent($form, $values, $presenter ? $this->template : null, $presenter ?: null));
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$presenter = $this->getPresenter();
		$this->em->beginTransaction();

		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var NavigationGroupText[] $texts */
			$texts = [];

			if ($this->groupId) {
				$group = $this->navigationGroupsService->get($this->groupId);
				$texts = $group->getTexts()->toArray();
			} else {
				$group = new NavigationGroup($values->type);
			}

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l]) && $v['title'])
					$texts[$l] = new NavigationGroupText($group, $l, $v['title']);

				if ($v['title'] == '' || $texts[$l]->title == '') {
					if ($texts[$l])
						$this->em->remove($texts[$l]);
					unset($texts[$l]);
					continue;
				}

				$texts[$l]->title = $v['title'];
				$this->em->persist($texts[$l]);
			}

			$group->setTexts($texts);
			$group->type   = $values->type;
			$group->locale = $this->translator->getLocale();

			$event                   = new FormSuccessEvent($form, $values, $this->template, $presenter);
			$event->custom['entity'] = $group;
			$this->eventDispatcher->dispatch(NavigationGroupForm::class . '::formSuccess', $event);

			$flashMessage = $group->getId() ? 'navigations.navigationGroupForm.edited' : 'navigations.navigationGroupForm.added';
			$this->em->persist($group);
			$this->em->flush();

			/** @var Navigation|null $navRoot */
			$navRoot = $this->em->getRepository(Navigation::class)
				->findOneBy(['group' => $group->getId(), 'lvl' => 0, 'parent' => null]);

			if (!$navRoot) {
				$navRoot = new Navigation($group, 'navigation.customLink', 1);
				$this->em->persist($navRoot);
				$this->em->flush();
			}

			$form->addCustomData('navigationGroupId', $group->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
			$this->em->commit();
		} catch (\Exception $e) {
			$this->em->rollback();
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ===========================  Get/Set
	 */

	public function setNavigationGroup(int $id): void
	{
		$this->groupId = $id;
		$this->group   = $this->navigationGroupsService->get($id);

		if (!$this->group)
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);

		$form  = $this['form'];
		$group = $this->group;

		$d = [
			'type' => $group->type,
		];

		foreach ($group->getTexts() as $text) {
			$d['title'][$text->getLang()] = $text->title;
		}

		$form->setDefaults($d);
	}
}
