<?php declare(strict_types = 1);

namespace Navigations\AdminModule\Presenters;

use Core\Model\UI\Form\BaseForm;
use Navigations\AdminModule\Components\INavigationFormFactory;
use Navigations\AdminModule\Components\INavigationGroupFormFactory;
use Navigations\AdminModule\Components\INavigationsGridFactory;
use Navigations\AdminModule\Components\INavigationsGroupsGridFactory;
use Navigations\AdminModule\Components\NavigationForm;
use Navigations\AdminModule\Components\NavigationGroupForm;
use Navigations\AdminModule\Components\NavigationsGrid;
use Navigations\AdminModule\Components\NavigationsGroupsGrid;
use Navigations\AdminModule\Model\Navigations;
use Navigations\AdminModule\Model\NavigationsGroups;

class DefaultPresenter extends BasePresenter
{
	/** @var Navigations @inject */
	public $navigationsService;

	/** @var NavigationsGroups @inject */
	public $navigationsGroupService;

	protected function startup()
	{
		parent::startup();
		$this['title']->setTitle($this->translator->translate('navigations.title.navigations'));
		$this->setHeader('navigations.title.navigations', 'fa fa-sitemap');
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault(): void
	{
		$this['navigation']->setData(['header' => [
			[
				'title' => 'navigations.menu.newItemNavigation',
				'link'  => $this->link('add!'),
				'ico'   => 'plus',
				'class' => 'ajax',
			], [
				'title' => 'navigations.menu.newNavigationGroup',
				'link'  => $this->link('addGroup!'),
				'ico'   => 'plus',
				'class' => 'ajax',
			],
		]]);
	}

	/*******************************************************************************************************************
	 * ============================== Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle       = $this->t('navigations.title.newNavigation');
		$this->template->modal            = 'navigationForm';
		$this->template->modalDialogClass = 'modal-xl';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$navigation = $this->navigationsService->get($id);

		if (!$navigation)
			$this->error();

		$this['navigationForm']->navId    = $id;
		$this->template->modalTitle       = $this->t('navigations.title.editNavigation',
			['title' => $navigation->getText() ? $navigation->getText()->getTitle() : '']);
		$this->template->modal            = 'navigationForm';
		$this->template->modalDialogClass = 'modal-xl';
		$this->redrawControl('modal');
	}

	public function handleAddGroup(): void
	{
		$this->template->modalTitle = $this->t('navigations.title.newNavigationGroup');
		$this->template->modal      = 'groupForm';
		$this->redrawControl('modal');
	}

	public function handleEditGroup(int $id): void
	{
		$group = $this->navigationsGroupService->get($id);

		if (!$group)
			$this->error();

		$this['navigationGroupForm']->groupId = $id;
		$this->template->modalTitle           = $this->t('navigations.title.editNavigationGroup',
			['title' => $group->getText() ? $group->getText()->title : '']);
		$this->template->modal                = 'groupForm';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentNavigationsGrid(INavigationsGridFactory $factory): NavigationsGrid
	{
		return $factory->create();
	}

	protected function createComponentNavigationForm(INavigationFormFactory $factory): NavigationForm
	{
		$control = $factory->create();

		if ($this->getParameter('id'))
			$control->setNavigation($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->handleEdit((int) $form->getCustomData('navigationId'));
			$this->redrawControl('flashes');
			$this['navigationsGrid']['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['navigationsGrid']['grid']->reload();
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentNavigationsGroupsGrid(INavigationsGroupsGridFactory $factory): NavigationsGroupsGrid
	{
		return $factory->create();
	}

	protected function createComponentNavigationGroupForm(INavigationGroupFormFactory $factory): NavigationGroupForm
	{
		$control = $factory->create();

		if ($this->getParameter('id'))
			$control->setNavigationGroup($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->handleEditGroup((int) $form->getCustomData('navigationGroupId'));
			$this->redrawControl('flashes');
			$this['navigationsGroupsGrid']['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['navigationsGroupsGrid']['grid']->reload();
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}
}
