<?php declare(strict_types = 1);

namespace Navigations\Model\Helper;

use Core\Components\Navigation\DaoNavigationItem;
use Kdyby\Doctrine\EntityManager;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Nette\Caching\IStorage;
use Nette\Localization\ITranslator;

class NavigationsHelper
{
	/** @var EntityManager */
	protected $em;

	/** @var IStorage */
	protected $cacheStorage;

	/** @var Cache */
	protected $cache;

	/** @var ITranslator */
	protected $translator;

	public function __construct(EntityManager $em, IStorage $IStorage, ITranslator $translator)
	{
		$this->em           = $em;
		$this->cacheStorage = $IStorage;
		$this->translator   = $translator;
	}

	protected function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, Navigations::CACHE_NAMESPACE);

		return $this->cache;
	}

	protected function getEr() { return $this->em->getRepository(Navigation::class); }

	/**
	 * @param string|null $lang
	 *
	 * @return DaoNavigationItem[]
	 */
	public function getPublished(?string $lang = null): array
	{
		$lang = $lang ?: $this->translator->getLocale();

		$navigations = $this->getEr()->createQueryBuilder('n')
			->select('n, nt, g, IDENTITY(n.parent) as parent')
			->join('n.texts', 'nt', 'WITH', 'nt.lang = :lang')
			->join('n.group', 'g')
			->where('nt.isPublished = 1')
			->andWhere('n.lvl > 0')
			->setParameter('lang', $lang)
			->orderBy('n.parent')->addOrderBy('n.lft')
			->getQuery()->getArrayResult();

		foreach ($navigations as $k => $v) {
			$arr              = $v[0];
			$arr['texts']     = $arr['texts'][$lang];
			$arr['parent']    = $v['parent'];
			$arr['groupType'] = $arr['group']['type'];
			$navigations[$k]  = $this->fillDao($arr);
		}

		return $navigations;
	}


	public function fillDao(array $item, ?DaoNavigationItem $parent = null): DaoNavigationItem
	{
		$dao                  = new DaoNavigationItem();
		$dao->id              = (int) $item['id'];
		$dao->title           = $item['texts']['title'];
		$dao->alias           = $item['texts']['alias'];
		$dao->isHomepage      = (int) $item['texts']['isHomepage'];
		$dao->groupType       = $item['group']['type'];
		$dao->componentType   = $item['componentType'];
		$dao->componentParams = $item['componentParams'];
		$dao->lang            = $item['texts']['lang'];
		$dao->seo             = $item['texts']['seo'];
		$dao->parentId        = (int) $item['parent'];
		$dao->isParent        = $dao->parentId ? true : false;
		$dao->class           = $item['params']['linkClass'] ?? '';
		$dao->lvl             = (int) $item['lvl'];
		$dao->setModified($item['modified'] ?: $item['created']);
		$dao->setParams($item['params'] ?: []);

		return $dao;
	}

	/**
	 * @param string|null $lang
	 *
	 * @return DaoNavigationItem[]
	 */
	public function getNavs(?string $lang = null): array
	{
		$lang = $lang ?: $this->translator->getLocale();

		return $this->getCache()->load('publishedNavs/' . $lang) ?: $this->getPublished();
	}
}
