<?php declare(strict_types = 1);

namespace Navigations\AdminModule\Components;

use Core\Model\Exceptions\EntityContainChildren;
use Core\Model\Lang\Langs;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use Navigations\AdminModule\Model\Navigations;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\NavigationsGroups;
use Nette\Utils\Html;

class NavigationsGrid extends BaseControl
{
	/** @var NavigationsGroups */
	protected $navigationsGroupsService;

	/** @var Navigations */
	protected $navigationsService;

	/** @var INavigationFormFactory */
	protected $formFactory;

	public function __construct(Navigations $navigations, NavigationsGroups $groups, INavigationFormFactory $formFactory)
	{
		$this->navigationsService       = $navigations;
		$this->navigationsGroupsService = $groups;
		$this->formFactory              = $formFactory;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleEdit(int $id): void
	{
		$presenter  = $this->getPresenter();
		$navigation = $this->navigationsService->get($id);

		if (!$navigation)
			$presenter->error();

		$this['navigationForm']->navId    = $id;
		$this->template->modalTitle       = $this->t('navigations.title.editNavigation',
			['title' => $navigation->getText() ? $navigation->getText()->getTitle() : '']);
		$this->template->modal            = 'navigationForm';
		$this->template->modalDialogClass = 'modal-xl';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$presenter = $this->getPresenter();

		try {
			if ($this->navigationsService->removeNavigation($id)) {
				$presenter->flashMessageSuccess('default.removed');
			} else {
				$presenter->flashMessageDanger('default.removeFailed');
			}
		} catch (EntityContainChildren $e) {
			$presenter->flashMessageDanger($e->getMessage());
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleSetHomepage(int $id, string $lang): void
	{
		$presenter = $this->getPresenter();

		if ($this->navigationsService->setHomepage($id, $lang)) {
			$presenter->flashMessageSuccess('navigations.navigation.homepageChanged');
		} else {
			$presenter->flashMessageDanger('navigations.navigation.homepageChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleInvertPublish(int $id, string $lang): void
	{
		$presenter = $this->getPresenter();

		if ($this->navigationsService->invertPublish($id, $lang)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$lang = $this->translator->getLocale();
		$grid = $this->createGrid();

		$qb = $this->navigationsService->getEr()->createQueryBuilder('n')
			->addSelect('nt, g, gt')
			->leftJoin('n.texts', 'nt')
			->join('n.group', 'g')
			->leftJoin('g.texts', 'gt', 'WITH', 'gt.lang = :lang')
			->where('n.lvl > 0')
			->setParameter('lang', $lang)
			->orderBy('g.id')->addOrderBy('n.lft');

		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('navigationsGrid:gridSortableRow!');

		// Columns
		$grid->addColumnLink('title', 'default.title', 'Default:edit')->setRenderer(function(Navigation $row) {
			$title = '';
			for ($i = 1; $i < $row->getLvl(); $i++)
				$title = '---' . $title;
			$title .= ' ' . ($row->getText() ? $row->getText()->getTitle() : ($row->getTexts()->first() ? $row->getTexts()->first()->getTitle() : ''));

			return Html::el('a', [
				'class' => 'ajax',
				'href'  => $this->link('edit!', $row->getId()),
			])->setText($title);
		});
		$grid->addColumnText('isHomepage', 'navigations.navigation.isHomepage')->setRenderer(function(Navigation $row) {
			return $this->gridRenderLangs($row, 'setHomepage!', 'isHomepage', 'btn-success', 'btn-outline-success');
		});
		$grid->addColumnText('alias', 'default.alias', 'text.alias');
		$grid->addColumnText('group', 'default.group', 'group.text.title');
		$grid->addColumnText('isPublished', 'default.isPublished')->setRenderer(function(Navigation $row) {
			return $this->gridRenderLangs($row, 'invertPublish!', 'isPublished', 'btn-success', 'btn-danger');
		});

		// Filter
		$groups = ['' => ''];
		foreach ($this->navigationsGroupsService->getAll($lang) as $group)
			$groups[$group->getId()] = $group->getText($lang)->title;

		$grid->addFilterText('title', 'default.title', 'nt.title')->setExactSearch();
		$grid->addFilterText('alias', 'default.alias', 'nt.alias')->setExactSearch();
		$grid->addFilterSelect('group', 'default.group', $groups, 'g.id');

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')->addClass('ajax');
		$grid->addAction('link','')->setIcon('link')->setBsType('info')->setRenderer(function (Navigation $row) {
			return Html::el('a', ['href' => $row->getBaseLink(), 'class' => 'btn btn-xs btn-info ', 'target' => '_blank'])->addHtml(Html::el('i', ['class' => 'fas fa-link']));
		});
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')
			->setConfirm('default.reallyDelete');

		// Columns prototype
		$grid->getColumn('isHomepage')->getElementPrototype('th')->class[]  = 'w1';
		$grid->getColumn('isPublished')->getElementPrototype('th')->class[] = 'w1';

		$grid->setRowCallback(function(Navigation $row, $tr) {
			$groupId = ($row->getGroup() ? $row->getGroup()->getId() : 0) . '-' .
				($row->getParent() ? $row->getParent()->getId() : '0') . '-' . $row->getLvl();
			/** @var Html $tr */
			$tr->addClass('group_' . $groupId);
			$tr->addAttributes(['data-group-tree' => $groupId]);

			return $tr;
		});

		return $grid;
	}

	protected function createComponentNavigationForm(): NavigationForm
	{
		$control = $this->formFactory->create();
		if ($this->getParameter('id'))
			$control->setNavigation((int) $this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	protected function gridRenderLangs(Navigation $row, string $handle, string $column, string $trueClass, string $falseClass): Html
	{
		$wrap = Html::el('div class=grid-langs');

		foreach ($this->langsService->getLangs(false) as $lang) {
			$text = $row->getText($lang->getTag());
			$el   = Html::el('div class=grid-langs__lang');
			if ($text) {
				$a = Html::el('a', [
					'class' => 'ajax btn btn-xs ' . ($text->$column ? $trueClass : $falseClass),
					'href'  => $this->link($handle, [$row->getId(), $lang->getTag()]),
				])->setText($lang->getShortTitle());
				$el->addHtml($a);
			}
			$wrap->addHtml($el);
		}

		return $wrap;
	}

	public function handleGridSortableRow(): void
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();
		$type      = $request->getPost('type');

		switch ($type) {
			case 'default':
				$id       = $request->getPost('id');
				$position = $request->getPost('position');

				if ($id != null && $position != null && $this->navigationsService->setPosition($id, $position))
					$presenter->flashMessageSuccess('default.positionChanged');
				else
					$presenter->flashMessageDanger('default.positionChangeFailed');
				break;
			case 'tree':
				$id   = $request->getPost('id');
				$move = $request->getPost('move');
				$er   = $this->navigationsService->getEr();
				$nav  = $this->navigationsService->get($id);

				if ($id && $move) {
					if ($move < 0)
						$er->moveUp($nav, abs($move));
					else if ($move > 0)
						$er->moveDown($nav, $move);
					$presenter->flashMessageSuccess('default.positionChanged');
				} else
					$presenter->flashMessageDanger('default.positionChangeFailed');

				$this->em->flush();
				$this->navigationsService->cleanCache();

				break;
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}
}
