<?php declare(strict_types = 1);

namespace Navigations\AdminModule\Components;

use Core\AdminModule\Model\Sites;
use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\FormValidateEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Navigations\AdminModule\Model\NavigationsGroups;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Entities\NavigationGroup;
use Navigations\Model\Entities\NavigationGroupText;
use Nette\Application\Attributes\Persistent;
use Nette\Utils\ArrayHash;
use Tracy\Debugger;

class NavigationGroupForm extends BaseControl
{
	#[Persistent]
	public ?int $groupId = null;

	/** @var NavigationsGroups @inject */
	public NavigationsGroups $navigationGroupsService;

	/** @var Sites @inject */
	public Sites $sitesService;

	public ?NavigationGroup $group = null;

	public function render(): void
	{
		$this->eventDispatcher->dispatch(new ComponentTemplateEvent($this->template, $this), self::class . '::render');
		$this->template->render($this->getTemplateFile() ?? __DIR__ . '/NavigationGroupForm.latte');
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('title', 'default.title')->setMaxLength(255)
			->setIsMultilanguage();
		$form->addText('type', 'default.type')->setMaxLength(255)->setRequired();

		$this->eventDispatcher->dispatch(
			new CreateFormEvent($form, $this->getPresenterIfExists() ? $this->template : null),
			self::class . '::createForm',
		);

		$form->addSaveCancelControl();

		$form->onValidate[] = $this->formValidate(...);
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$presenter = $this->presenter;

		$nameFilled = false;
		foreach ($values->title as $v) {
			if ($v) {
				$nameFilled = true;
			}
		}

		if (!$nameFilled) {
			$form->getComponent('title')->addError('default.formMessages.required');
		}

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}

		$this->eventDispatcher->dispatch(
			new FormValidateEvent($form, $values, $this->template, $presenter),
			self::class . '::validateForm',
		);
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$presenter = $this->presenter;
		$this->em->beginTransaction();

		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var NavigationGroupText[] $texts */
			$texts = [];

			if ($this->groupId) {
				$group = $this->navigationGroupsService->get($this->groupId);
				$texts = $group->getTexts()->toArray();
			} else {
				$group = new NavigationGroup($values->type);
			}

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l]) && $v['title']) {
					$texts[$l] = new NavigationGroupText($group, $l, $v['title']);
				}

				if ($texts[$l] && ($v['title'] == '' || $texts[$l]->title == '')) {
					$this->em->remove($texts[$l]);
					unset($texts[$l]);
				}

				if (!$texts[$l]) {
					continue;
				}

				$texts[$l]->title = $v['title'];
				$this->em->persist($texts[$l]);
			}

			$group->setTexts($texts);
			$group->type   = $values->type;
			$group->locale = $this->translator->getLocale();

			$event                   = new FormSuccessEvent($form, $values, $this->template, $presenter);
			$event->custom['entity'] = $group;
			$this->eventDispatcher->dispatch($event, self::class . '::formSuccess');

			$flashMessage = $group->getId(
			) ? 'navigations.navigationGroupForm.edited' : 'navigations.navigationGroupForm.added';
			$this->em->persist($group);
			$this->em->flush();

			/** @var Navigation|null $navRoot */
			$navRoot = $this->em->getRepository(Navigation::class)
				->findOneBy(['group' => $group->getId(), 'lvl' => 0, 'parent' => null]);

			if (!$navRoot) {
				foreach ($this->sitesService->getAll() as $site) {
					$navRoot = new Navigation($group, 'navigation.customLink', $site);
					$this->em->persist($navRoot);
				}
				$this->em->flush();
			}

			$form->addCustomData('navigationGroupId', $group->getId());
			$this->presenter->flashMessageSuccess($flashMessage);
			$this->em->commit();
		} catch (Exception $e) {
			Debugger::log($e);
			$this->em->rollback();
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ===========================  Get/Set
	 */

	public function setNavigationGroup(int $id): void
	{
		$this->groupId = $id;
		$this->group   = $this->navigationGroupsService->get($id);

		if (!$this->group) {
			$this->presenter->error();
		}

		$form  = $this['form'];
		$group = $this->group;

		$d = [
			'type' => $group->type,
		];

		foreach ($group->getTexts() as $text) {
			$d['title'][$text->getLang()] = $text->title;
		}

		$form->setDefaults($d);
	}

}
