<?php declare(strict_types = 1);

namespace Navigations\Model\Helper;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Contributte\Translation\Translator;
use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use Doctrine\ORM\EntityRepository;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\NavigationConfig;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Http\Request;
use Nette\Utils\Strings;
use Throwable;
use function str_ends_with;

class NavigationsHelper
{
	protected ?Cache $cache = null;

	protected array $cPublished = [];

	protected array $cacheDep = [
		Cache::Tags   => [Navigations::CACHE_NAMESPACE],
		Cache::Expire => '1 week',
	];

	public function __construct(
		protected EntityManagerDecorator $em,
		protected Storage                $cacheStorage,
		protected Translator             $translator,
		protected Sites                  $sitesService,
		protected Request                $httpRequest,
	)
	{
	}

	protected function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, Navigations::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	protected function getEr(): EntityRepository
	{
		return $this->em->getRepository(Navigation::class);
	}

	public function getSitesService(): Sites { return $this->sitesService; }

	/**
	 * @return DaoNavigationItem[]
	 * @throws Throwable
	 */
	public function getPublished(?string $lang = null): array
	{
		$lang      = $lang ?: $this->translator->getLocale();
		$siteIdent = $this->sitesService->getCurrentSite()->getIdent();

		$key = $siteIdent . '-' . $lang;

		if (!isset($this->cPublished[$key])) {
			$this->cPublished[$key] = $this->getCache()->load('published_' . $key, function(&$dep) use (
				$siteIdent,
				$lang,
			) {
				$dep  = $this->cacheDep;
				$data = [];

				foreach ($this->getEr()->createQueryBuilder('n')
					         ->select('n, nt, g, IDENTITY(n.parent) as parent')
					         ->join('n.texts', 'nt', 'WITH', 'nt.lang = :lang AND nt.isPublished = 1')
					         ->join('n.group', 'g')
					         ->andWhere('n.lvl > 0')
					         ->andWhere('n.site = :site')
					         ->setParameters(new ArrayCollection([new Parameter('lang', $lang), new Parameter('site', $siteIdent)]))
					         ->orderBy('n.parent')->addOrderBy('n.lft')
					         ->getQuery()->getArrayResult() as $k => $v) {
					$arr              = $v[0];
					$arr['texts']     = $arr['texts'][$lang];
					$arr['parent']    = $v['parent'];
					$arr['groupType'] = $arr['group']['type'];

					$data[] = $this->fillDao($arr);
				}

				return $data;
			});
		}

		return $this->cPublished[$key];
	}

	public function fillDao(array $item, ?DaoNavigationItem $parent = null): DaoNavigationItem
	{
		$dao                    = new DaoNavigationItem();
		$dao->id                = (int) $item['id'];
		$dao->title             = $item['texts']['title'];
		$dao->alias             = $item['texts']['alias'];
		$dao->isHomepage        = (int) $item['texts']['isHomepage'];
		$dao->groupType         = $item['group']['type'];
		$dao->componentType     = $item['componentType'];
		$dao->componentParams   = (array) $item['componentParams'];
		$dao->ico               = $item['icon'];
		$dao->icoSecondary      = $item['iconSecondary'];
		$dao->hideInMobileMenu  = (int) $item['texts']['hideInMobileMenu'];
		$dao->hideInDesktopMenu = (int) $item['texts']['hideInDesktopMenu'];
		$dao->lang              = $item['texts']['lang'];
		$dao->description       = $item['texts']['description'];
		$dao->img               = $item['texts']['img'];
		$dao->seo               = $item['texts']['seo'];
		$dao->customText1       = $item['texts']['customText1'];
		$dao->customText2       = $item['texts']['customText2'];
		$dao->parentId          = (int) $item['parent'];
		$dao->isParent          = $dao->parentId ? true : false;
		$dao->class             = $item['params']['linkClass'] ?? '';
		$dao->lvl               = (int) $item['lvl'];
		$dao->setModified($item['modified'] ?: $item['created']);
		$dao->setParams($item['params'] ?: []);

		if (isset($item['params']['openIn'])) {
			$dao->openIn = $item['params']['openIn'];
			unset($item['params']['openIn']);
		}

		if ($item['texts']['fullUrl']) {
			$dao->link             = '/' . $item['texts']['fullUrl'];
			$dao->isFullLinkFilled = true;
		}

		return $dao;
	}

	/**
	 * @return DaoNavigationItem[]
	 * @throws Throwable
	 */
	public function getNavs(?string $lang = null): array
	{
		$lang = $lang ?: $this->translator->getLocale();

		return $this->getCache()->load('publishedNavs/' . $lang) ?: $this->getPublished($lang);
	}

	public function getHttpRequest(): Request { return $this->httpRequest; }

	public static function removeUrlSuffix(string $url): string
	{
		$endSuffix = (string) NavigationConfig::loadScalar('urlSuffix');
		if ($endSuffix && str_ends_with($url, (string) $endSuffix)) {
			return Strings::substring($url, 0, -strlen((string) $endSuffix));
		}

		return $url;
	}
}
