<?php declare(strict_types = 1);

namespace Navigations\AdminModule\Components;

use Core\Model\Helpers\Arrays;
use Doctrine\ORM\QueryBuilder;
use Core\Model\Exceptions\EntityContainChildren;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Core\Model\UI\Form\BaseForm;
use Navigations\AdminModule\Model\Navigations;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Entities\NavigationGroup;
use Navigations\Model\Entities\NavigationText;
use Navigations\AdminModule\Model\NavigationsGroups;
use Nette\Utils\Html;

class NavigationsGrid extends BaseControl
{
	protected string                 $siteIdent;
	protected NavigationsGroups      $adminNavigationsGroups;
	protected Navigations            $navigationsService;
	protected INavigationFormFactory $formFactory;

	public function __construct(
		string                 $siteIdent,
		Navigations            $navigations,
		INavigationFormFactory $formFactory,
		NavigationsGroups      $adminNavigationsGroups
	)
	{
		$this->siteIdent              = $siteIdent;
		$this->navigationsService     = $navigations;
		$this->formFactory            = $formFactory;
		$this->adminNavigationsGroups = $adminNavigationsGroups;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleEdit(int $id): void
	{
		$presenter  = $this->presenter;
		$navigation = $this->navigationsService->get($id);

		if (!$navigation) {
			$presenter->error();
		}

		$this['navigationForm']->navId    = $id;
		$this->template->modalTitle       = $this->t('navigations.title.editNavigation',
			['title' => $navigation->getText() ? $navigation->getText()->getTitle() : '']);
		$this->template->modal            = 'navigationForm';
		$this->template->modalDialogClass = 'modal-xl';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$presenter = $this->presenter;

		try {
			if ($this->navigationsService->removeNavigation($id)) {
				$presenter->flashMessageSuccess('default.removed');
			} else {
				$presenter->flashMessageDanger('default.removeFailed');
			}
		} catch (EntityContainChildren $e) {
			$presenter->flashMessageDanger($e->getMessage());
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleSetHomepage(int $id, string $lang): void
	{
		$presenter = $this->presenter;

		if ($this->navigationsService->setHomepage($id, $lang)) {
			$presenter->flashMessageSuccess('navigations.navigation.homepageChanged');
		} else {
			$presenter->flashMessageDanger('navigations.navigation.homepageChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleInvertPublish(int $id, string $lang): void
	{
		$presenter = $this->presenter;

		if ($this->navigationsService->invertPublish($id, $lang)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$lang = $this->translator->getLocale();
		$grid = $this->createGrid();

		$qb = $this->navigationsService->getEr()->createQueryBuilder('n')
			->addSelect('nt, g, gt')
			->leftJoin('n.texts', 'nt')
			->join('n.group', 'g')
			->leftJoin('g.texts', 'gt', 'WITH', 'gt.lang = :lang')
			->where('n.lvl > 0')
			->andWhere('n.site = :site')
			->setParameters([
				'lang' => $lang,
				'site' => $this->siteIdent,
			])
			->orderBy('g.id')->addOrderBy('n.lft');

		$grid->setDataSource($qb);
		$grid->setItemsPerPageList([100, 200, 300], true);

		$grid->setSortable();
		$grid->setSortableHandler('navigationsGrid-' . $this->siteIdent . ':gridSortableRow!');

		// Columns
		$grid->addColumnLink('title', 'default.title', 'Default:edit')->setRenderer(function(Navigation $row) {
			$title = '';
			for ($i = 1; $i < $row->getLvl(); $i++) {
				$title = '---' . $title;
			}
			$title .= ' ' . ($row->getText() ? $row->getText()->getTitle() : ($row->getTexts()->first() ? $row->getTexts()->first()->getTitle() : ''));

			return Html::el('a', [
				'class' => 'ajax',
				'href'  => $this->link('edit!', $row->getId()),
			])->setText($title);
		});
		$grid->addColumnText('isHomepage', 'navigations.navigation.isHomepage')->setRenderer(function(Navigation $row) {
			return $this->gridRenderLangs($row, 'setHomepage!', 'isHomepage', 'btn-success', 'btn-outline-success');
		});
		$grid->addColumnText('alias', 'default.alias', 'text.alias');
		$grid->addColumnText('group', 'default.group', 'group.text.title');
		$grid->addColumnText('isPublished', 'default.isPublished')->setRenderer(function(Navigation $row) {
			return $this->gridRenderLangs($row, 'invertPublish!', 'isPublished', 'btn-success', 'btn-danger');
		});

		// Filter
		$groups = ['' => ''] + $this->adminNavigationsGroups->getOptionsForSelect();

		$grid->addFilterText('title', 'default.title')->setCondition(function(QueryBuilder $qb, $value) {
			if (!Arrays::contains($qb->getAllAliases(), 'ntFilter')) {
				$qb->innerJoin('n.texts', 'ntFilter');
			}

			$qb->andWhere('ntFilter.title LIKE :title')->setParameter('title', '%' . $value . '%');
		});
		$grid->addFilterText('alias', 'default.alias', 'nt.alias')->setCondition(function(QueryBuilder $qb, $value) {
			if (!Arrays::contains($qb->getAllAliases(), 'ntFilter')) {
				$qb->innerJoin('n.texts', 'ntFilter');
			}

			$qb->andWhere('ntFilter.alias LIKE :alias')->setParameter('alias', '%' . $value . '%');
		});
		$grid->addFilterSelect('group', 'default.group', $groups, 'g.id');

		// Actions
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')->addClass('ajax')
			->setConfirm('default.reallyDelete');

		// Columns prototype
		$grid->getColumn('isHomepage')->getElementPrototype('th')->addClass('w1');
		$grid->getColumn('isPublished')->getElementPrototype('th')->addClass('w1');

		$grid->setRowCallback(function(Navigation $row, $tr) {
			$groupId = $row->getGroup()->getId() . '-' .
				($row->getParent() ? $row->getParent()->getId() : '0') . '-' . $row->getLvl();
			/** @var Html $tr */
			$tr->addClass('group_' . $groupId);
			$tr->addAttributes(['data-group-tree' => $groupId]);

			return $tr;
		});

		return $grid;
	}

	protected function createComponentNavigationForm(): NavigationForm
	{
		$control = $this->formFactory->create();
		if ($this->getParameter('id')) {
			$control->setNavigation((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form) {
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->presenter->payload->hideModal = true;
			$this->presenter->redrawControl('flashes');
			$this['grid']->reload();
		};
		$control['form']->getComponent('saveControl')->closeModalOnCancel();

		return $control;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	protected function gridRenderLangs(Navigation $row, string $handle, string $column, string $trueClass, string $falseClass): Html
	{
		$wrap = Html::el('div class=grid-langs');

		foreach ($this->langsService->getLangs(false) as $lang) {
			//			$text = $this->texts[$row->getId()][$lang->getTag()] ?? null;
			$text = $row->getText($lang->getTag());
			$el   = Html::el('div class=grid-langs__lang');
			if ($text) {
				$a = Html::el('a', [
					'class' => 'ajax btn btn-xs ' . ($text->$column ? $trueClass : $falseClass),
					'href'  => $this->link($handle, [$row->getId(), $lang->getTag()]),
				])->setText($lang->getShortTitle());
				$el->addHtml($a);
			}
			$wrap->addHtml($el);
		}

		return $wrap;
	}

	public function handleGridSortableRow(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();
		$type      = $request->getPost('type');

		if ($type === 'tree') {
			$id   = $request->getPost('id');
			$move = $request->getPost('move');
			$er   = $this->navigationsService->getEr();
			$nav  = $this->navigationsService->get($id);

			if ($id && $move) {
				if ($move < 0) {
					$er->moveUp($nav, (int) abs($move));
				} else if ($move > 0) {
					$er->moveDown($nav, $move);
				}
				$presenter->flashMessageSuccess('default.positionChanged');
			} else {
				$presenter->flashMessageDanger('default.positionChangeFailed');
			}

			$this->em->flush();
			$this->navigationsService->cleanCache();
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}
}
