<?php declare(strict_types = 1);

namespace Navigations\FrontModule\Components;

use Core\Components\Navigation\DaoNavigationItem;
use Core\FrontModule\Presenters\BasePresenter;
use Core\Model\Helpers\CoreHelper;
use Core\Model\Helpers\Strings;
use Core\Model\Parameters;
use Core\Model\Sites;
use Core\Model\UI\BaseControl;
use Currency\FrontModule\Components\CurrencySwitch;
use Navigations\FrontModule\Model\LinksHelper;
use Nette\Utils\Json;
use ReflectionException;

/**
 * @method BasePresenter getPresenter($throw = true)
 */
class LangSwitcher extends BaseControl
{
	public const CURRENCY_MAP = [
		'sk' => 'EUR',
		'en' => 'EUR',
		'cs' => 'CZK',
		'de' => 'EUR',
	];

	protected LinksHelper $linksHelper;
	protected Sites       $sites;

	public function __construct(
		LinksHelper $linksHelper,
		Sites       $sites
	)
	{
		$this->linksHelper = $linksHelper;
		$this->sites       = $sites;
	}

	/**
	 * @param array $args
	 *
	 * @throws ReflectionException
	 */
	public function render($args = []): void
	{
		$links = $this->getLinks();

		$this->template->renderArgs = $args;
		$this->template->links      = $links;
		$this->template->activeLang = $links[$this->translator->getLocale()];
		$this->template->render($this->getTemplateFile());
	}

	public function handleChange(string $url): void
	{
		$userAgent = $_SERVER['HTTP_USER_AGENT'] ?? null;
		$sessionId = $_COOKIE['PHPSESSID'] ?? null;
		$ts        = time();
		$secret    = Parameters::load('system.transferToken');

		$data = [
			'sid' => $sessionId,
			'ua'  => substr(sha1($userAgent), 0, 10),
			'ts'  => $ts,
			'url' => $url,
			'ip'  => CoreHelper::ipPrefix($_SERVER['REMOTE_ADDR'] ?? ''),
		];

		$data['sig'] = hash_hmac('sha256', Json::encode($data), $secret);

		$hash = rtrim(strtr(base64_encode(Json::encode($data)), '+/', '-_'), '=');

		$target = $url . (Strings::contains($url, '?') ? '&' : '?') . 'transfer=' . $hash;

		$this->presenter->redirectUrl($target);
	}

	public function handleChangeDefault(): void
	{
		$url = $this->getPresenter()->getHttpRequest()->getUrl();

		$this->presenter->redirectUrl($url->getBaseUrl() . $url->getRelativePath());
	}

	public function getLinks(): array
	{
		$presenter = $this->getPresenter();
		$links     = [];

		$domain    = $this->sites->getCurrentSite()->getCurrentDomain();
		$domainUrl = $domain ? $domain->getDomain() : null;

		foreach ($this->linksHelper->getOtherLangLinks($presenter->getActiveNavigation(), $presenter->getParameters(), $presenter->getName()) as $lang => $link) {
			if (class_exists('Currency\FrontModule\Components\CurrencySwitch')) {
				/** @var CurrencySwitch $currencySwitch */
				$currencySwitch = $this->getPresenter()->getComponent('currencySwitch', false);

				if ($currencySwitch && $link) {
					foreach (self::CURRENCY_MAP as $l => $c) {
						if ($lang === $l && isset($currencySwitch->getActiveCurrencies()[$c])) {
							$link['link'] .= (Strings::contains((string) $link['link'], '?') ? '&' : '?') .
								'curr=' . $currencySwitch->getActiveCurrencies()[$c]->getCode() . '&do=langSwitcher-changeDefault';
							break;
						}
					}
				}
			}

			if ($domainUrl && !Strings::startsWith((string) $link['link'], 'https://' . $domainUrl)) {
				/** @var DaoNavigationItem|null $nav */
				$nav = array_key_exists('navigation', $link) ? $link['navigation'] : null;
				if ($link['link']) {
					$link['link'] = $this->link('change!', ['url' => $link['link']]);
				} else if ($nav && is_string($nav->link)) {
					$link['link'] = $nav->link;
				}
			}

			$links[$lang] = $link;
		}

		return $links;
	}
}
