<?php declare(strict_types = 1);

namespace Navigations\Model\Entities;

use Core\Model\Entities\Site;
use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Traits\TExtraField;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Core\Model\Entities\TId;
use Nette\Utils\DateTime;

/**
 * @Gedmo\Tree(type="nested")
 * @ORM\Table(name="navigations__navigation", indexes={@ORM\Index(name="type_idx", columns={"type"})})
 * @ORM\Entity(repositoryClass="Core\Model\Entities\Repository\NestedTreeRepository")
 * @ORM\EntityListeners({"NavigationListener", "Core\Model\Entities\TranslateListener"})
 */
class Navigation
{
	use TId;
	use TTranslateListener;
	use TExtraField;

	public const EXTRA_FIELD_SECTION = 'navigation';

	/**
	 * @var Collection<string, NavigationText>
	 * @ORM\OneToMany(targetEntity="NavigationText", mappedBy="navigation", indexBy="lang")
	 */
	public Collection $texts;

	/**
	 * @ORM\Column(type="string", nullable=false, length=255)
	 */
	public string $link;

	/**
	 * @ORM\Column(name="type", type="string", nullable=false)
	 */
	public string $componentType;

	/**
	 * @ORM\Column(type="array", nullable=true)
	 */
	public ?array $componentParams = [];

	/**
	 * @ORM\Column(name="attribs", type="array", nullable=true)
	 */
	private ?array $params = [];

	/**
	 * @ORM\ManyToOne(targetEntity="NavigationGroup")
	 * @ORM\JoinColumn(name="group_id", referencedColumnName="id", nullable=false, onDelete="RESTRICT")
	 */
	private NavigationGroup $group;

	/**
	 * @var Collection<Navigation>
	 * @ORM\OneToMany(targetEntity="Navigation", mappedBy="parent")
	 * @ORM\OrderBy({"lft" = "ASC"})
	 */
	private Collection $children;

	/**
	 * @Gedmo\TreeParent
	 * @ORM\ManyToOne(targetEntity="Navigation", inversedBy="childrens")
	 * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", onDelete="RESTRICT")
	 */
	private ?Navigation $parent = null;

	/**
	 * @Gedmo\TreeLeft
	 * @ORM\Column(type="integer")
	 */
	private int $lft;

	/**
	 * @Gedmo\TreeLevel
	 * @ORM\Column(type="integer")
	 */
	private int $lvl;

	/**
	 * @Gedmo\TreeRight
	 * @ORM\Column(type="integer")
	 */
	private int $rgt;

	/**
	 * @Gedmo\TreeRoot
	 * @ORM\ManyToOne(targetEntity="Navigation")
	 * @ORM\JoinColumn(name="tree_root", referencedColumnName="id", nullable=true, onDelete="CASCADE")
	 */
	private ?Navigation $root = null;

	/**
	 * @var DateTime|null
	 * @Gedmo\Timestampable(on="update")
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	private $modified;

	/**
	 * @var DateTime|null
	 * @Gedmo\Timestampable(on="create")
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	private $created;

	/**
	 * @ORM\ManyToOne(targetEntity="Core\Model\Entities\Site")
	 * @ORM\JoinColumn(name="site_id", referencedColumnName="ident", onDelete="CASCADE", nullable=false)
	 */
	protected Site $site;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $icon = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $iconSecondary = null;

	public function __construct(NavigationGroup $group, string $type, Site $site)
	{
		$this->group         = $group;
		$this->componentType = $type;
		$this->site          = $site;
		$this->children      = new ArrayCollection;
		$this->texts         = new ArrayCollection;
		$this->link          = '';
	}

	public function getParams(): array { return $this->params ?: []; }

	/**
	 * @return mixed|null $key
	 */
	public function getParam(string $key) { return $this->params[$key] ?? null; }

	/**
	 * @param mixed|null $value
	 */
	public function setParam(string $key, $value): self
	{
		if ($value) {
			$this->params[$key] = $value;
		} else {
			unset($this->params[$key]);
		}

		return $this;
	}

	public function setTexts(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	public function getText(?string $lang = null): ?NavigationText { return $this->texts[$lang ?: $this->locale] ?? null; }

	/** @return Collection<string, NavigationText> */
	public function getTexts() { return $this->texts; }

	public function getGroup(): NavigationGroup { return $this->group; }

	public function setGroup(NavigationGroup $group): self
	{
		$this->group = $group;

		return $this;
	}

	public function getParent(): ?Navigation { return $this->parent; }

	public function setParent(?Navigation $parent = null): self
	{
		$this->parent = $parent;

		return $this;
	}

	/** @return Collection<Navigation> */
	public function getChildren() { return $this->children; }

	public function getRoot(): Navigation { return $this->root; }

	public function getLft(): int { return $this->lft; }

	public function getRgt(): int { return $this->rgt; }

	public function getLvl(): int { return $this->lvl; }

	public function getModified(): DateTime { return $this->modified; }

	public function getCreated(): DateTime { return $this->created; }

	public function setSite(Site $site): self
	{
		$this->site = $site;

		return $this;
	}

	public function getSite(): Site { return $this->site; }
}
