<?php declare(strict_types = 1);

namespace Navigations\Model\Entities;

use Core\Model\Entities\Redirect;
use Core\AdminModule\Model\Redirects;
use Doctrine;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Nette\Caching\Storage;
use Nette\Utils\Arrays;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Nette\SmartObject;

class NavigationTextListener implements EventSubscriberInterface
{
	use SmartObject;

	private Cache         $cache;
	protected Redirects   $redirectsService;
	protected Navigations $navigationsService;

	public function __construct(Storage $cacheStorage, Redirects $redirects, Navigations $navigations)
	{
		$this->cache              = new Cache($cacheStorage, Navigations::CACHE_NAMESPACE);
		$this->redirectsService   = $redirects;
		$this->navigationsService = $navigations;
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	/**
	 * @ORM\PostUpdate
	 */
	public function postUpdateHandler(NavigationText $navigationText, LifecycleEventArgs $event): void
	{
		$this->cache->clean([
			Cache::TAGS => [Navigations::CACHE_NAMESPACE],
		]);

		$em         = $event->getEntityManager();
		$changeSet  = $em->getUnitOfWork()->getEntityChangeSet($navigationText);
		$navigation = $navigationText->navigation;

		if (isset($changeSet['alias']) && !$navigationText->isHomepage) {
			[$from, $to] = $changeSet['alias'];
			$path = $this->navigationsService->getEr()->getPath($navigation);
			array_shift($path);
			array_pop($path);

			if ($path) {
				$tFrom = [];
				$tTo   = [];
				foreach ($path as $p) {
					$tFrom[] = $p->getText($navigationText->getLang())->getAlias();
					$tTo[]   = $p->getText($navigationText->getLang())->getAlias();
				}
				$tFrom[] = $from;
				$tTo[]   = $to;
				$from    = implode('/', $tFrom);
				$to      = implode('/', $tTo);
			}

			$fromExist = $this->redirectsService->getEr()->findOneBy([
				'from'        => $to, 'package' => 'Navigations',
				'relationKey' => 'Navigation',
			]);
			if ($fromExist) {
				$em->remove($fromExist);
			}

			/** @var Redirect[] $toExist */
			$toExist = $this->redirectsService->getEr()->findBy([
				'to'          => $from, 'package' => 'Navigations',
				'relationKey' => 'Navigation',
			]);
			foreach ($toExist as $red) {
				$red->to = $to;
				$em->persist($red);
			}

			// TODO přesunout do služby?
			$redirect                = new Redirect('Navigation redirect - ' . $from . ' -> ' . $to, $from, $to);
			$redirect->isLocked      = 1;
			$redirect->package       = Arrays::first(explode('\\', __NAMESPACE__));
			$redirect->relationKey   = 'Navigation';
			$redirect->relationValue = $navigation->getId() ? ((string) $navigation->getId()) : null;

			$em->persist($redirect);
			$em->flush();
		}
	}
}
