<?php declare(strict_types = 1);

namespace Navigations\AdminModule\Components;

use Core\AdminModule\Model\Sites;
use Core\FrontModule\Model\SeoContainer;
use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\FormValidateEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\CheckboxListInput;
use Core\Model\UI\Form\Controls\SelectInput;
use Doctrine\ORM\ORMException;
use Exception;
use Navigations\AdminModule\Model\Navigations;
use Navigations\AdminModule\Model\NavigationsGroups;
use Navigations\Model\BaseItem;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Entities\NavigationText;
use Navigations\Model\ItemsCollector;
use Navigations\Model\Providers\INavigationItem;
use Nette\Application\UI\Presenter;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Forms\Controls\HiddenField;
use Nette\Forms\Controls\SelectBox;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use Nette\Utils\Json;
use Nette\Utils\Strings;
use Nette\Utils\Validators;

class NavigationForm extends BaseControl
{
	/** @var int|null @persistent */
	public ?int               $navId      = null;
	public ?Navigation        $navigation = null;
	protected INavigationItem $component;

	public function __construct(
		protected ItemsCollector    $itemsCollector,
		protected NavigationsGroups $navigationsGroups,
		protected Navigations       $navigationsService,
		protected SeoContainer      $seoContainerService,
		protected Sites             $sitesService,
		Storage                     $storage,
	)
	{
		$this->cache = new Cache($storage, Navigations::CACHE_NAMESPACE);
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];
		$this->eventDispatcher->dispatch(new ComponentTemplateEvent($this->template, $this), self::class . '::render');
		$this->template->render($this->getTemplateFile());
	}

	/**
	 * @param Presenter $presenter
	 *
	 * @throws ORMException
	 */
	protected function attached($presenter): void
	{
		parent::attached($presenter);

		/** @var Request $httpRequest */
		$httpRequest = $presenter->getHttpRequest();

		$site         = $httpRequest->getPost('site') ?? $this['form']->getComponent('site')->getValue();
		$navGroupPost = $httpRequest->getPost('navigationGroup') ?? $this['form']->getComponent('navigationGroup')
			->getValue();
		$this->loadParentItems($site, $navGroupPost);
	}

	public function handleComponentCustom(string $componentId): void
	{
		$component = $this->itemsCollector->getItemById($componentId);

		$data = ['componentId' => $componentId] + $this->getPost('componentData', []);
		$component->navigationFormCustomHandle($this, $data);
	}

	/**
	 * @throws ORMException
	 */
	protected function loadParentItems(string $site, int|string|null $groupId): void
	{
		/** @var Navigation|null $root */
		$root = $this->navigationsService->getEr()->findOneBy([
			'group'  => $groupId,
			'parent' => null,
			'lvl'    => 0,
			'site'   => $site,
		]);

		if (!$root) {
			$root = new Navigation(
				$this->navigationsGroups->getReference($groupId),
				'navigation.customLink',
				$this->sitesService->getReference($site),
			);
			$this->em->persist($root);
			$this->em->flush();
		}

		$parents = [$root->getId() => ' '];
		$q       = $this->navigationsService->getEr()->createQueryBuilder('n')
			->select('n.id, nt.title, n.lvl')
			->join('n.texts', 'nt', 'WITH', 'nt.lang = :lang')
			->where('n.lvl > 0')
			->andWhere('n.group = :gId')
			->andWhere('n.site = :site')
			->orderBy('n.root')->addOrderBy('n.lft')
			->setParameters([
				'site' => $site,
				'gId'  => $groupId,
				'lang' => $this->translator->getLocale(),
			]);

		if ($this->navigation) {
			$q->andWhere('n.id != :curId')->setParameter('curId', $this->navigation->getId());
		}
		foreach ($q->getQuery()->getScalarResult() as $v) {
			$title = ' ' . $v['title'];
			for ($i = 1; $i < $v['lvl']; $i++) {
				$title = '---' . $title;
			}
			$parents[$v['id']] = $title;
		}
		$this['form']->getComponent('parent')->setItems($parents);
	}

	public function loadComponent(int|string $componentId): void
	{
		$form = $this['form'];

		if ($this->getPresenterIfExists()) {
			$post = $this->presenter->getHttpRequest()->getPost();

			if ($post) {
				$form->addCustomData('loadComponent', $post);
			}
		}

		if ($form->getComponent('component', false)) {
			$form->removeComponent($form['component']);
		}

		$component = $this->itemsCollector->getItemById((string) $componentId);

		if ($component) {
			$this->component = $component;
			$this->setComponentToTemplate($component);
			$container = $component->getFormContainer($form);
			$form->addComponent($container, 'component');
			$component = $form->getComponent('component');
			if ($this->navigation) {
				foreach ($component->components as $k => $c) {
					$value = $this->navigation->componentParams[$k] ?? null;

					if (!$value) {
						continue;
					}

					if ($c instanceof SelectInput) {
						$c->setDefaultValue($value);
					} else if ($c instanceof CheckboxListInput) {
						$tmp = [];
						foreach ($value as $v) {
							if (array_key_exists($v, $c->getItems())) {
								$tmp[] = $v;
							}
						}

						$c->setDefaultValue($tmp);
					} else {
						$c->setDefaultValue($value);
					}
				}
			}
		}
	}

	protected function setComponentToTemplate(INavigationItem $component): void
	{
		if ($this->lookup(null, false)) {
			$this->template->component = $component;
		} else {
			$this->onAnchor[] = function() use ($component) {
				$this->template->component = $component;
			};
		}
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleLoadInputs(string $componentId): void
	{
		$presenter = $this->presenter;

		try {
			$this->loadComponent($componentId);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (Exception) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('component');
		$presenter->redrawControl('flashes');
	}

	/**
	 * @param string $site
	 */
	public function handleLoadParents($site, int|string|null $group): void
	{
		$presenter = $this->presenter;

		try {
			$this->loadParentItems($site, $group);
			$presenter->flashMessageSuccess('default.loaded');
		} catch (Exception) {
			$presenter->flashMessageDanger('default.error');
		}

		$this->redrawControl('parent');
		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$components = ['' => $this->t('default.choose')];
		foreach ($this->itemsCollector->getItems() as $group) {
			$arr = [];

			foreach ($group['items'] as $class) {
				/** @var BaseItem $class */
				$arr[$class->getId()] = $this->t($class->getTitle());
			}

			$components[$this->t($group['title'])] = $arr;
		}

		$navGroups = [];
		foreach ($this->navigationsGroups->getAll() as $group) {
			$navGroups[$group->getId()] = $group->getText()->title;
		}

		$openIn = [
			'default' => 'navigations.openIn.default',
			'_blank'  => 'navigations.openIn.blank',
		];

		$form->addText('title', 'navigations.navigation.title')->setMaxLength(255)
			->setIsMultilanguage();
		$form->addText('alias', 'navigations.navigation.alias')->setMaxLength(255)
			->setIsMultilanguage();
		$form->addBool('isPublished', 'default.isPublished')
			->setIsMultilanguage()->setDefaultValue(1);
		$form->addSelect('openIn', 'navigations.navigation.openIn', $openIn)->setDefaultValue('default');
		$form->addText('fullUrl', 'navigations.navigation.fullUrl')
			->setIsMultilanguage();

		$sites = $this->sitesService->getOptionsForSelect();
		if (count($sites) > 1) {
			$form->addSelect('site', 'navigations.navigation.site', $sites)
				->setRequired();
		} else {
			$form->addHidden('site');
		}
		$form->getComponent('site')->setDefaultValue(array_values($sites)[0]);

		$form->addSelect('navigationGroup', 'navigations.navigation.navigationGroup', $navGroups)
			->setDefaultValue(array_keys($navGroups)[0])
			->setRequired();
		$form->addSelect('parent', $this->t('default.parent'), [])->setTranslator(null);
		$form->addText('customText1', 'navigations.navigation.customText1')
			->setIsMultilanguage();
		$form->addText('customText2', 'navigations.navigation.customText2')
			->setIsMultilanguage();
		$form->addFilesManager('icon', 'navigations.navigation.icon')
			->setNullable();
		$form->addFilesManager('iconSecondary', 'navigations.navigation.iconSecondary')
			->setNullable();
		$form->addText('linkClass', 'navigations.navigation.linkClass');
		$form->addText('pageClass', 'navigations.navigation.pageClass');
		$form->addFilesManager('img', 'navigations.navigation.image')->setIsMultilanguage();
		$form->addTextArea('description', 'navigations.navigation.description')->setIsMultilanguage();
		$form->addSelect('componentType', $this->t('navigations.navigation.component'), $components)
			->setTranslator(null)->setRequired();

		$form->addComponent($this->seoContainerService->getContainer(true), 'seo');

		$this->eventDispatcher->dispatch(
			new CreateFormEvent($form, $this->getPresenterIfExists() ? $this->template : null),
			self::class . '::createForm',
		);

		$form->addSaveCancelControl();

		$form->onValidate[] = $this->formValidate(...);
		$form->onSuccess[]  = $this->formSuccess(...);

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$presenter = $this->presenter;
		$nav       = $this->navId ? $this->navigationsService->get($this->navId) : null;

		$titleEmpty = true;
		foreach ($values->title as $l => $v) {
			if ($v !== '') {
				$titleEmpty = false;
				break;
			}
		}

		foreach ($values->alias as $l => $v) {
			if (!$v && $values->title[$l]) {
				$values->alias[$l] = Strings::webalize($values->title[$l]);
			}
			$texts = $nav ? $nav->getTexts() : null;

			if (!$nav || (isset($texts[$l]) && $texts[$l]->getAlias() != $v)) {
				$tmpNav = $this->navigationsService->findByAlias($v, $l, $values->parent);
				if ($tmpNav) {
					$form->addError($this->t('navigations.navigationForm.aliasExists') . ' (' . $l . ')');
				}
			}
		}

		if ($titleEmpty) {
			$form->addError(
				$this->t(
					'default.formMessages.requiredField',
					['input' => $this->t($form->getComponent('title')->caption)],
				),
				false,
			);
		}

		if ($form->hasErrors()) {
			if ($form->isAjax()) {
				$this->redrawControl('form');
			} else {
				$this->redirect('this');
			}
		}

		$this->eventDispatcher->dispatch(
			new FormValidateEvent($form, $values, $this->template, $presenter),
			self::class . '::validateForm',
		);
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$presenter = $this->presenter;
		$this->em->beginTransaction();

		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var NavigationText[] $texts */
			$texts    = [];
			$navGroup = $this->navigationsGroups->getReference($values->navigationGroup);

			if ($this->navId) {
				$navigation = $this->navigationsService->get($this->navId);
				$texts      = $navigation->getTexts()->toArray();
			} else {
				$navigation = new Navigation(
					$navGroup,
					$values->componentType,
					$this->sitesService->getReference($values->site)
				);
				$this->em->persist($navigation);
				$this->em->flush($navigation);
			}
			$currentSite = $navigation->getSite();

			foreach ($langValues as $l => $v) {
				if (!isset($texts[$l]) && $v['title']) {
					$texts[$l] = new NavigationText(
						$navigation,
						$v['title'],
						$v['description'] ?: null,
						$l,
						$v['img'] ?: null
					);
				}

				if ($v['title'] == '' || $texts[$l]->getTitle() == '') {
					if (!Validators::isNone($texts[$l])) {
						$this->em->remove($texts[$l]);
					}
					unset($texts[$l]);
					continue;
				}

				$texts[$l]->setTitle($v['title'])
					->setAlias($v['alias'])
					->setDescription($v['description'])
					->setImg($v['img'])
					->setSeo($v['seo'] ?? [])
					->setLang($l);

				$texts[$l]->isPublished = $v['isPublished'] ? 1 : 0;
				$texts[$l]->customText1 = $v['customText1'] ?: null;
				$texts[$l]->customText2 = $v['customText2'] ?: null;
				$texts[$l]->fullUrl     = $v['fullUrl'] ?: null;

				$this->em->persist($texts[$l]);
			}

			$navigation->setTexts($texts);
			$navigation->setGroup($navGroup);
			$navigation->setSite($this->sitesService->getReference($values->site));
			$navigation->link            = '';
			$navigation->componentType   = $values->componentType;
			$navigation->componentParams = $this->presenter->getRequest()->getPost('component');
			$navigation->locale          = $this->translator->getLocale();
			$navigation->icon            = $values->icon ?: null;
			$navigation->iconSecondary   = $values->iconSecondary ?: null;

			$navigation->setParam('linkClass', $values->linkClass);
			$navigation->setParam('pageClass', $values->pageClass);
			$navigation->setParam('openIn', $values->openIn == 'default' ? null : $values->openIn);
			$navigation->setParent($values->parent ? $this->navigationsService->getReference($values->parent) : null);
			$navigation->setGroup(
				$values->navigationGroup ? $this->navigationsGroups->getReference($values->navigationGroup) : null,
			);

			$event                   = new FormSuccessEvent($form, $values, $this->template, $presenter);
			$event->custom['entity'] = $navigation;
			$this->eventDispatcher->dispatch($event, self::class . '::formSuccess');

			$flashMessage = $navigation->getId(
			) ? 'navigations.navigationForm.edited' : 'navigations.navigationForm.added';
			$this->em->persist($navigation);
			$this->em->flush();
			$form->addCustomData('navigationId', $navigation->getId());
			$form->addCustomData('navigationCurrentSite', $currentSite->getIdent());
			$form->addCustomData('navigationSite', $navigation->getSite()->getIdent());
			$this->presenter->flashMessageSuccess($flashMessage);
			$this->em->commit();
			$this->cacheStorage->clean([Cache::Tags => ['eshopNavigation', 'navigation']]);
			$this->cache->clean([Cache::Tags => [Navigations::CACHE_NAMESPACE]]);
		} catch (Exception $e) {
			$this->em->rollback();
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ===========================  Get/Set
	 */

	public function setNavigation(int $id): void
	{
		$this->navId      = $id;
		$this->navigation = $this->navigationsService->get($id);

		if (!$this->navigation) {
			$this->presenter->error();
		}

		$form = $this['form'];
		$n    = $this->navigation;
		$this->loadParentItems($n->getSite()->getIdent(), $n->getGroup()->getId());

		$d = [
			'icon'          => $n->icon,
			'iconSecondary' => $n->iconSecondary,
		];

		$openIn = $n->getParam('openIn');
		if ($openIn && array_key_exists($openIn, $form->getComponent('openIn')->getItems())) {
			$d['openIn'] = $openIn;
		}
		if ($form['site'] instanceof SelectBox && array_key_exists(
				$n->getSite()
					->getIdent(),
				$form->getComponent('site')->getItems(),
			)) {
			$d['site'] = $n->getSite()->getIdent();
		} else if ($form['site'] instanceof HiddenField) {
			$d['site'] = $n->getSite()->getIdent();
		}
		if (array_key_exists($n->getGroup()->getId(), $form->getComponent('navigationGroup')->getItems())) {
			$d['navigationGroup'] = $n->getGroup()->getId();
		}
		if ($n->getParent() && array_key_exists($n->getParent()->getId(), $form->getComponent('parent')->getItems())) {
			$d['parent'] = $n->getParent()->getId();
		}
		if ($n->componentType && array_key_exists(
				$n->componentType,
				$form->getComponent('componentType')
					->getItems(),
			)) {
			$d['componentType']                                = $n->componentType;
			$form->getComponent('componentType')
				->getControlPrototype()->attrs['data-default'] = Json::encode($n->componentParams);
		}

		foreach ($n->getParams() as $k => $v) {
			if (isset($form[$k])) {
				$d[$k] = $v;
			}
		}

		$form->setDefaults($d);

		if ($n->componentType) {
			$this->loadComponent($n->componentType);
		}

		$d = [];
		foreach ($n->getTexts() as $l => $text) {
			$d['title'][$l]       = $text->getTitle();
			$d['img'][$l]         = $text->getImg();
			$d['description'][$l] = $text->getDescription();
			$d['alias'][$l]       = $text->getAlias();
			$d['isPublished'][$l] = $text->isPublished;
			$d['customText1'][$l] = $text->customText1;
			$d['customText2'][$l] = $text->customText2;
			$d['fullUrl'][$l]     = $text->fullUrl;
		}
		$this->seoContainerService->setDefaultsFromEntity($form->getComponent('seo'), $n->getTexts()->toArray());

		$form->setDefaults($d);

		$this->eventDispatcher->dispatch(
			new SetFormDataEvent($form, $this->navigation),
			self::class . '::setNavigation',
		);
	}
}
