<?php declare(strict_types = 1);

namespace Navigations\Model\Entities;

use Core\Model\Entities\Repository\NestedTreeRepository;
use Core\Model\Entities\Site;
use Core\Model\Entities\TId;
use Core\Model\Entities\TranslateListener;
use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Traits\TExtraField;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;

#[ORM\Table(name: 'navigations__navigation')]
#[ORM\Index(columns: ['type'], name: 'type_idx')]
#[Gedmo\Tree(type: 'nested')]
#[ORM\Entity(repositoryClass: NestedTreeRepository::class)]
#[ORM\EntityListeners([NavigationListener::class, TranslateListener::class])]
class Navigation
{
	use TId;
	use TTranslateListener;
	use TExtraField;

	public const EXTRA_FIELD_SECTION = 'navigation';

	/**
	 * @var Collection<string, NavigationText>
	 */
	#[ORM\OneToMany(mappedBy: 'navigation', targetEntity: NavigationText::class, indexBy: 'lang')]
	public Collection $texts;

	#[ORM\Column(type: Types::STRING, length: 255, nullable: false)]
	public string $link;

	#[ORM\Column(name: 'type', type: Types::STRING, nullable: false)]
	public string $componentType;

	#[ORM\Column(type: Types::ARRAY, nullable: true)]
	public ?array $componentParams = [];

	#[ORM\Column(name: 'attribs', type: Types::ARRAY, nullable: true)]
	private ?array $params = [];

	#[ORM\ManyToOne(targetEntity: NavigationGroup::class)]
	#[ORM\JoinColumn(name: 'group_id', referencedColumnName: 'id', nullable: false, onDelete: 'RESTRICT')]
	private NavigationGroup $group;

	/**
	 * @var Collection<Navigation>
	 */
	#[ORM\OneToMany(mappedBy: 'parent', targetEntity: Navigation::class)]
	#[ORM\OrderBy(['lft' => 'ASC'])]
	private Collection $children;

	#[Gedmo\TreeParent]
	#[ORM\ManyToOne(targetEntity: Navigation::class, inversedBy: 'childrens')]
	#[ORM\JoinColumn(name: 'parent_id', referencedColumnName: 'id', onDelete: 'RESTRICT')]
	private ?Navigation $parent = null;

	#[Gedmo\TreeLeft]
	#[ORM\Column(type: Types::INTEGER)]
	private int $lft;

	#[Gedmo\TreeLevel]
	#[ORM\Column(type: Types::INTEGER)]
	private int $lvl;

	#[Gedmo\TreeRight]
	#[ORM\Column(type: Types::INTEGER)]
	private int $rgt;

	#[Gedmo\TreeRoot]
	#[ORM\ManyToOne(targetEntity: Navigation::class)]
	#[ORM\JoinColumn(name: 'tree_root', referencedColumnName: 'id', nullable: true, onDelete: 'CASCADE')]
	private ?Navigation $root = null;

	#[Gedmo\Timestampable(on: 'update')]
	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	protected ?DateTimeInterface $modified = null;

	#[Gedmo\Timestampable(on: 'create')]
	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	protected ?DateTimeInterface $created = null;

	#[ORM\ManyToOne(targetEntity: Site::class)]
	#[ORM\JoinColumn(name: 'site_id', referencedColumnName: 'ident', nullable: false, onDelete: 'CASCADE')]
	protected Site $site;

	#[ORM\Column(type: Types::STRING, length: 255, nullable: true)]
	public ?string $icon = null;

	#[ORM\Column(type: Types::STRING, length: 255, nullable: true)]
	public ?string $iconSecondary = null;

	public function __construct(NavigationGroup $group, string $type, Site $site)
	{
		$this->group         = $group;
		$this->componentType = $type;
		$this->site          = $site;
		$this->children      = new ArrayCollection;
		$this->texts         = new ArrayCollection;
		$this->link          = '';
	}

	public function getParams(): array { return $this->params ?: []; }

	/**
	 * @return mixed|null $key
	 */
	public function getParam(string $key) { return $this->params[$key] ?? null; }

	/**
	 * @param mixed|null $value
	 */
	public function setParam(string $key, $value): self
	{
		if ($value) {
			$this->params[$key] = $value;
		} else {
			unset($this->params[$key]);
		}

		return $this;
	}

	public function setTexts(array $texts): self
	{
		$this->texts = new ArrayCollection($texts);

		return $this;
	}

	public function getText(?string $lang = null): ?NavigationText
	{
		return $this->texts[$lang ?: $this->locale] ?? null;
	}

	/** @return Collection<string, NavigationText> */
	public function getTexts() { return $this->texts; }

	public function getGroup(): NavigationGroup { return $this->group; }

	public function setGroup(NavigationGroup $group): self
	{
		$this->group = $group;

		return $this;
	}

	public function getParent(): ?Navigation { return $this->parent; }

	public function setParent(?Navigation $parent = null): self
	{
		$this->parent = $parent;

		return $this;
	}

	/** @return Collection<Navigation> */
	public function getChildren() { return $this->children; }

	public function getRoot(): Navigation { return $this->root; }

	public function getLft(): int { return $this->lft; }

	public function getRgt(): int { return $this->rgt; }

	public function getLvl(): int { return $this->lvl; }

	public function getModified(): DateTimeInterface { return $this->modified; }

	public function getCreated(): DateTimeInterface { return $this->created; }

	public function setSite(Site $site): self
	{
		$this->site = $site;

		return $this;
	}

	public function getSite(): Site { return $this->site; }
}
