<?php declare(strict_types = 1);

namespace Navigations\Model\Entities;

use Core\AdminModule\Model\Redirects;
use Core\Model\Entities\Redirect;
use Doctrine;
use Doctrine\ORM\Event\PostUpdateEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\ORM\PersistentCollection;
use Navigations\Model\Navigations;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\SmartObject;
use Nette\Utils\Arrays;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class NavigationTextListener implements EventSubscriberInterface
{
	use SmartObject;

	protected Cache $cache;

	public function __construct(
		protected Storage     $cacheStorage,
		protected Redirects   $redirects,
		protected Navigations $navigations,
	)
	{
		$this->cache = new Cache($cacheStorage, Navigations::CACHE_NAMESPACE);
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	#[ORM\PostUpdate]
	public function postUpdateHandler(NavigationText $navigationText, PostUpdateEventArgs $event): void
	{
		$this->cache->clean([
			Cache::Tags => [Navigations::CACHE_NAMESPACE],
		]);

		$em         = $event->getObjectManager();
		$changeSet  = $em->getUnitOfWork()->getEntityChangeSet($navigationText);
		$navigation = $navigationText->navigation;

		if (isset($changeSet['alias']) && !$navigationText->isHomepage) {
			[$from, $to] = $changeSet['alias'] instanceof PersistentCollection ? $changeSet['alias']->toArray()[0] : $changeSet['alias'];
			$path = $this->navigations->getEr()->getPath($navigation);
			array_shift($path);
			array_pop($path);

			if ($path) {
				$tFrom = [];
				$tTo   = [];
				foreach ($path as $p) {
					$tFrom[] = $p->getText($navigationText->getLang())->getAlias();
					$tTo[]   = $p->getText($navigationText->getLang())->getAlias();
				}
				$tFrom[] = $from;
				$tTo[]   = $to;
				$from    = implode('/', $tFrom);
				$to      = implode('/', $tTo);
			}

			$fromExist = $this->redirects->getEr()->findOneBy([
				'from'        => $to, 'package' => 'Navigations',
				'relationKey' => 'Navigation',
			]);
			if ($fromExist) {
				$em->remove($fromExist);
			}

			/** @var Redirect[] $toExist */
			$toExist = $this->redirects->getEr()->findBy([
				'to'          => $from, 'package' => 'Navigations',
				'relationKey' => 'Navigation',
			]);
			foreach ($toExist as $red) {
				$red->to = $to;
				$em->persist($red);
			}

			// TODO přesunout do služby?
			$redirect                = new Redirect('Navigation redirect - ' . $from . ' -> ' . $to, $from, $to);
			$redirect->isLocked      = 1;
			$redirect->package       = Arrays::first(explode('\\', __NAMESPACE__));
			$redirect->relationKey   = 'Navigation';
			$redirect->relationValue = $navigation->getId() ? ((string) $navigation->getId()) : null;

			$em->persist($redirect);
			$em->flush();
		}
	}
}
