<?php declare(strict_types = 1);

namespace Navigations\AdminModule\Model;

use Contributte\Translation\Translator;
use Core\Model\Entities\Repository\NestedTreeRepository;
use Core\Model\Exceptions\EntityContainChildren;
use Core\Model\Helpers\BaseEntityService;
use Doctrine\ORM\Query\Expr\Join;
use Exception;
use Navigations\Model\Entities\Navigation;
use Navigations\Model\Entities\NavigationText;
use Nette\Caching\Cache;

/**
 * @method Navigation|null getReference($id)
 * @method Navigation[] getAll()
 * @method Navigation|null get($id)
 * @method NestedTreeRepository getEr()
 */
class Navigations extends BaseEntityService
{
	public const CACHE_NAMESPACE = 'navigations';

	/** @var string */
	protected $entityClass = Navigation::class;

	protected string $entityClassText = NavigationText::class;
	protected ?array $cOptionsForSelect = null;
	protected Translator $translator;

	public function __construct(Translator $translator)
	{
		$this->translator = $translator;
	}

	public function getCache(): Cache
	{
		if ($this->cache === null) {
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		}

		return $this->cache;
	}

	public function cleanCache(): void
	{
		$this->cacheStorage->clean([Cache::TAGS => ['eshopNavigation', 'navigation']]);
		$this->getCache()->clean([Cache::TAGS => [self::CACHE_NAMESPACE]]);
	}

	public function setHomepage(int $id, string $lang): bool
	{
		$nav = $this->get($id);

		if (!$nav || !$nav->getText($lang)) {
			return false;
		}

		$this->em->beginTransaction();
		try {
			$oldHpText = $this->em->getRepository($this->entityClassText)->createQueryBuilder('nt')
				->innerJoin('nt.navigation', 'n')
				->where('nt.isHomepage = 1')
				->andWhere('nt.lang = :lang')
				->andWhere('n.site = :site')
				->setParameters([
					'lang' => $lang,
					'site' => $nav->getSite()->getIdent(),
				])
				->setMaxResults(1)
				->getQuery()->getOneOrNullResult();

			if ($oldHpText) {
				$oldHpText->isHomepage = 0;
				$this->em->persist($oldHpText);
			}

			$nav->getText($lang)->isHomepage = 1;
			$this->em->persist($nav->getText($lang))->flush();
			$this->em->commit();
		} catch (Exception $e) {
			$this->em->rollback();

			return false;
		}

		return true;
	}

	public function invertPublish(int $id, string $lang): bool
	{
		$nav = $this->get($id);

		if (!$nav || !$nav->getText($lang)) {
			return false;
		}

		$nav->getText($lang)->isPublished = (int) !$nav->getText($lang)->isPublished;
		$this->em->persist($nav->getText($lang))->flush();

		$this->cacheStorage->clean([Cache::TAGS => ['navigation']]);
		$cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);
		$cache->clean([Cache::TAGS => [self::CACHE_NAMESPACE]]);

		return true;
	}

	public function removeNavigation(int $id): bool
	{
		if ($nav = $this->get($id)) {
			if ($nav->getChildren()->count()) {
				throw new EntityContainChildren('navigations.navigation.removeFailedHasChilds');
			}
			$this->em->remove($nav);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/**
	 * @return Navigation[]
	 */
	public function findByAlias(string $alias, ?string $lang = null, ?int $compareParent = null): array
	{
		$qb = $this->getEr()->createQueryBuilder('n')
			->andWhere('nt.alias = :alias')
			->setParameter('alias', $alias);

		if ($lang) {
			$qb->join('n.texts', 'nt', 'WITH', 'nt.lang = :lang')
				->setParameter('lang', $lang);
		} else {
			$qb->join('n.texts', 'nt');
		}

		if ($compareParent) {
			$qb->andWhere('n.parent = :parent')
				->setParameter('parent', $compareParent);
		}

		return $qb->getQuery()->getResult();
	}

	/**
	 * @return array
	 * @throws Exception
	 */
	public function fixTreeStructure(): array
	{
		$result = ['status' => 'error'];

		return $result;

		//		$groups  = $this->em->getRepository(NavigationGroup::class)->createQueryBuilder('ng')->select('ng.title, ng.id')->getQuery()->getArrayResult();
		//		$parents = [];
		//		/** @var Navigation[] $navs */
		//		$navs = $this->getEr()->createQueryBuilder('n')->orderBy('n.parent')->addOrderBy('n.position', 'DESC')->getQuery()->getResult();
		//		/** @var Navigation[] $roots */
		//		$roots = [];
		//
		//		// Vytvoření dočasného rootu
		//		$tmp     = $this->em->getRepository(NavigationGroup::class)->findOneBy([]);
		//		$tmpRoot = new Navigation('ROOT - TMP', $this->em->getReference(NavigationGroup::class, $tmp->getId()), '', 'navigation.customLink', 0);
		//		$this->em->persist($tmpRoot);
		//		$this->em->flush();
		//
		//		// Získání všech rootů
		//		foreach ($groups as $g) {
		//			$root = $this->getEr()->findOneBy(['title' => 'ROOT - ' . $g['title']]);
		//			if (!$root) {
		//				$root = new Navigation('ROOT - ' . $g['title'], $this->em->getReference(NavigationGroup::class, $g['id']), '', 'navigation.customLink', 0);
		//				$this->em->persist($root);
		//				$this->em->flush();
		//			}
		//			$this->getEr()->createQueryBuilder('n')->update()
		//				->set('n.root', ':root')
		//				->where('n.id = :id')
		//				->getQuery()->execute(['root' => $root->getId(), 'id' => $root->getId()]);
		//			$roots[$g['id']] = $root;
		//		}
		//
		//		// Získání všech navigací bez rootu
		//		foreach ($navs as $k => $n) {
		//			if ($roots[$n->group->getId()]->getId() == $n->getId() || strpos($n->title, 'ROOT - ') === 0) {
		//				unset($navs[$k]);
		//				continue;
		//			}
		//
		//			if ($n->getParent())
		//				$parents[$n->getId()] = $n->getParent();
		//		}
		//
		//		// Vložení do dočasného rootu
		//		foreach ($navs as $n) {
		//			$this->getEr()->persistAsFirstChildOf($n, $tmpRoot);
		//			$this->em->flush();
		//		}
		//
		//		$this->getEr()->verify();
		//
		//		// Vložení do správných rootů
		//		foreach ($navs as $n) {
		//			$this->getEr()->persistAsFirstChildOf($n, $roots[$n->group->getId()]);
		//			$this->em->flush();
		//		}
		//
		//		// Nastavení nadřazených
		//		foreach ($navs as $n) {
		//			if (isset($parents[$n->getId()])) {
		//				$this->getEr()->persistAsFirstChildOf($n, $parents[$n->getId()]);
		//				$this->em->flush();
		//			}
		//		}
		//
		//		$errors = $this->getEr()->verify();
		//		$this->em->flush();
		//
		//		$this->em->remove($tmpRoot);
		//		$tmpRoots = $this->getEr()->findBy(['title' => 'ROOT - TMP']);
		//		if ($tmpRoots)
		//			$this->em->remove($tmpRoots);
		//		$this->em->flush();
		//
		//		if (is_array($errors))
		//			$result['errors'] = $errors;
		//		else
		//			$result['status'] = 'ok';
		//
		//		return $result;
	}

	public function getOptionsForSelect(): array
	{
		if ($this->cOptionsForSelect === null) {
			$this->cOptionsForSelect = [];

			$bySite = [];

			foreach ($this->getEr()->createQueryBuilder('n')
				         ->select('n.id, nt.title, n.lvl, ngt.title as groupTitle, IDENTITY(n.site) as site')
				         ->innerJoin('n.texts', 'nt', Join::WITH, 'nt.lang = :lang')
				         ->innerJoin('n.group', 'ng')
				         ->innerJoin('ng.texts', 'ngt', Join::WITH, 'ngt.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->orderBy('n.root')->addOrderBy('n.lft')
				         ->getQuery()->getResult() as $v) {
				$title = str_repeat('---', $v['lvl'] - 1) . ' ' . $v['title'];

				$bySite[$v['site']][$v['groupTitle']][$v['id']] = trim($title);
			}

			if (count($bySite) > 1) {
				foreach ($bySite as $site => $navsByGroup) {
					foreach ($navsByGroup as $group => $navs) {
						$this->cOptionsForSelect[$site . ' - ' . $group] = $navs;
					}
				}
			} else {
				$this->cOptionsForSelect = isset(array_values($bySite)[0]) ? array_values($bySite)[0] : [];
			}

			$bySite = null;
		}

		return $this->cOptionsForSelect ?: [];
	}

}